// file      : examples/cxx/tree/multiroot/dom-parse.cxx
// author    : Boris Kolpackov <boris@codesynthesis.com>
// copyright : not copyrighted - public domain

#include "dom-parse.hxx"

#include <xercesc/dom/DOM.hpp>
#include <xercesc/util/XMLUniDefs.hpp> // chLatin_*
#include <xercesc/framework/Wrapper4InputSource.hpp>

#include <xsd/cxx/xml/sax/std-input-source.hxx>
#include <xsd/cxx/xml/dom/bits/error-handler-proxy.hxx>

#include <xsd/cxx/tree/exceptions.hxx>
#include <xsd/cxx/tree/error-handler.hxx>

using namespace xercesc;
namespace xml = xsd::cxx::xml;
namespace tree = xsd::cxx::tree;

xml::dom::auto_ptr<DOMDocument>
parse (std::istream& is, const std::string& id, bool validate)
{
  const XMLCh ls_id [] = {chLatin_L, chLatin_S, chNull};

  // Get an implementation of the Load-Store (LS) interface.
  //
  DOMImplementation* impl (
    DOMImplementationRegistry::getDOMImplementation (ls_id));

  // Create a DOMBuilder.
  //
  xml::dom::auto_ptr<DOMBuilder> parser (
    impl->createDOMBuilder(DOMImplementationLS::MODE_SYNCHRONOUS, 0));

  // Discard comment nodes in the document.
  //
  parser->setFeature (XMLUni::fgDOMComments, false);

  // Enable datatype normalization.
  //
  parser->setFeature (XMLUni::fgDOMDatatypeNormalization, true);

  // Do not create EntityReference nodes in the DOM tree. No
  // EntityReference nodes will be created, only the nodes
  // corresponding to their fully expanded substitution text
  // will be created.
  //
  parser->setFeature (XMLUni::fgDOMEntities, false);

  // Perform Namespace processing.
  //
  parser->setFeature (XMLUni::fgDOMNamespaces, true);

  // Do not include ignorable whitespace in the DOM tree.
  //
  parser->setFeature (XMLUni::fgDOMWhitespaceInElementContent, false);

  // Enable/Disable validation.
  //
  parser->setFeature (XMLUni::fgDOMValidation, validate);
  parser->setFeature (XMLUni::fgXercesSchema, validate);
  parser->setFeature (XMLUni::fgXercesSchemaFullChecking, false);

  // We will release the DOM document ourselves.
  //
  parser->setFeature (XMLUni::fgXercesUserAdoptsDOMDocument, true);

  // Set error handler.
  //
  tree::error_handler<char> eh;
  xml::dom::bits::error_handler_proxy<char> ehp (eh);
  parser->setErrorHandler (&ehp);

  // Prepare input stream.
  //
  xml::sax::std_input_source isrc (is, id);
  Wrapper4InputSource wrap (&isrc, false);

  xml::dom::auto_ptr<DOMDocument> doc (parser->parse (wrap));

  eh.throw_if_failed<tree::parsing<char> > ();

  return doc;
}
