// file      : examples/cxx/tree/dbxml/driver.cxx
// author    : Boris Kolpackov <boris@codesynthesis.com>
// copyright : not copyrighted - public domain

#include <memory>   // std::auto_ptr
#include <string>
#include <cassert>
#include <iostream>

#include <dbxml/DbXml.hpp>

#include "library.hxx"

using std::cerr;
using std::endl;
using std::string;
using std::auto_ptr;

using namespace DbXml;
using namespace xsd::cxx; // for xml::string

void
print_document (const string& desc,
                XmlContainer container,
                const string& name)
{
  XmlDocument doc (container.getDocument (name));

  string content;
  doc.getContent (content);

  cerr << endl
       << desc << endl
       << content << endl;
}

int
main ()
{
  try
  {
    using namespace library;

    XmlManager manager;

    {
      XmlContainer container (manager.createContainer ("new.bdbxml"));

      XmlUpdateContext update_context (manager.createUpdateContext ());

      XmlQueryContext context (manager.createQueryContext ());
      context.setNamespace ("lib", "http://www.codesynthesis.com/library");


      // Create a new document from an object representation.
      //
      {
        // Create a new catalog with one book.
        //
        catalog c;

        book b (20530902,                        // ISBN
                title ("The Elements of Style"), // Title
                genre::reference,                // Genre
                "ES");                           // ID

        author strunk ("William Strunk, Jr.", "1869-07-01");
        strunk.died ("1946-09-26");

        b.author ().push_back (strunk);
        c.book ().push_back (b);


        // Create a new XML document.
        //
        XmlDocument doc (manager.createDocument ());
        doc.setName ("new.xml");


        // Obtain its DOM representation and add the root element.
        //
        xercesc::DOMDocument& dom_doc (*doc.getContentAsDOM ());

        dom_doc.appendChild (
          dom_doc.createElementNS (
            xml::string ("http://www.codesynthesis.com/library").c_str (),
            xml::string ("lib:catalog").c_str ()));


        // Serialize the object representation to the XML document.
        //
        catalog_ (dom_doc, c);


        // Place the document into the container.
        //
        container.putDocument (doc, update_context);

        print_document ("after create:", container, "new.xml");
      }

      // Create an object representation from a document in DB.
      //
      {
        // Resolve the document in the container.
        //
        XmlDocument doc (container.getDocument ("new.xml"));


        // Create the object representation from the document's DOM.
        //
        auto_ptr<catalog> c (catalog_ (*doc.getContentAsDOM ()));

        cerr << *c << endl;
      }


      // Lookup a document fragment.
      //

      string query ("collection('new.bdbxml')/lib:catalog/book[@id='ES']");

      // Find "The Elements of Style".
      //
      XmlValue v;
      XmlResults results (manager.query (query, context));

      if (results.next (v))
      {
        // Create an object representation from the document fragment.
        //
        auto_ptr<book> b (
          new book (
            *static_cast<xercesc::DOMElement*> (v.asNode ())));

        cerr << *b << endl;


        // Add another author, change the availability status.
        //
        author white ("E.B. White", "1899-07-11");
        white.died ("1985-10-01");

        b->author ().push_back (white);
        b->available (false);


        // Update the document fragment from the object representation.
        //
        *static_cast<xercesc::DOMElement*> (v.asNode ()) << *b;


        // Update the document in the container.
        //
        XmlDocument doc (v.asDocument ());
        container.updateDocument (doc, update_context);
      }

      print_document ("after update:", container, "new.xml");
    }

    manager.removeContainer ("new.bdbxml");
  }
  catch (const std::exception& e)
  {
    cerr << e.what () << endl;
    return 1;
  }
}
