// file      : examples/cxx/parser/mixed/driver.cxx
// author    : Boris Kolpackov <boris@codesynthesis.com>
// copyright : not copyrighted - public domain

#include <string>
#include <vector>
#include <iostream>

#include "text.hxx"

using std::cout;
using std::cerr;
using std::endl;
using std::string;


struct anchor
{
  anchor (const string& text, const string& uri)
      : uri_ (uri), text_ (text)
  {
  }

  const string&
  text () const
  {
    return text_;
  }

  const string&
  uri () const
  {
    return uri_;
  }

private:
  string uri_;
  string text_;
};


// We would like to re-use the string parser in implementation
// of the anchor parser. Since the return types of our post()
// functions are not covariant, we will need to "rename" the
// anchor::post function before we can mix the two implementation
// together. For more information see the mix-in example.
//
struct anchor_parser_base: parsers::anchor<anchor, string>
{
  virtual ::anchor
  post_anchor () = 0;

  virtual ::anchor
  post ()
  {
    return post_anchor ();
  }
};

struct anchor_parser: anchor_parser_base, xml_schema::string
{
  virtual void
  href (const std::string& uri)
  {
    uri_ = uri;
  }

  virtual ::anchor
  post_anchor ()
  {
    return ::anchor (xml_schema::string::post (), uri_);
  }

private:
  std::string uri_;
};


struct text_parser: parsers::text<void, anchor>
{
  virtual void
  a (const anchor& a)
  {
    cout << a.text () << "[" << anchors_.size () << "]";
    anchors_.push_back (a);
  }

  virtual void
  _any_characters (const xml_schema::ro_string& s)
  {
    cout << s;
  }

  virtual void
  post ()
  {
    for (anchors::const_iterator i (anchors_.begin ());
         i != anchors_.end ();
         ++i)
    {
      cout << "[" << i - anchors_.begin () << "] " << i->uri () << endl;
    }
  }

private:
  typedef std::vector<anchor> anchors;
  anchors anchors_;
};


int
main (int argc, char* argv[])
{
  if (argc != 2)
  {
    cerr << "usage: " << argv[0] << " text.xml" << endl;
    return 1;
  }

  try
  {
    // Construct the parser.
    //
    xml_schema::string string_p;
    anchor_parser anchor_p;
    text_parser text_p;

    anchor_p.href_parser (string_p);
    text_p.a_parser (anchor_p);

    // Parse the XML instance document.
    //
    xml_schema::document<void> doc_p (text_p, "text");

    doc_p.parse (argv[1]);
  }
  catch (const xml_schema::exception& e)
  {
    cerr << e << endl;
    return 1;
  }
  catch (const std::ios_base::failure&)
  {
    cerr << "io failure" << endl;
    return 1;
  }
}
