// file      : xsd/cxx/parser/expat/elements.txx
// author    : Boris Kolpackov <boris@codesynthesis.com>
// copyright : Copyright (c) 2005-2006 Code Synthesis Tools CC
// license   : GNU GPL v2 + exceptions; see accompanying LICENSE file

#include <istream>
#include <fstream>

#include <xsd/cxx/parser/error-handler.hxx>

namespace xsd
{
  namespace cxx
  {
    namespace parser
    {
      namespace expat
      {
        // event_router
        //

        template <typename C>
        event_router<C>::
        event_router (event_consumer<C>& consumer)
            : consumer_ (consumer)
        {
        }

        // Expat thunks.
        //
        template <typename C>
        void XMLCALL event_router<C>::
        start_element (
          void* data, XML_Char const* ns_name, XML_Char const** atts)
        {
          event_router& r (*reinterpret_cast<event_router*> (data));
          r.start_element_ (ns_name, atts);
        }

        template <typename C>
        void XMLCALL event_router<C>::
        end_element (void* data, XML_Char const* ns_name)
        {
          event_router& r (*reinterpret_cast<event_router*> (data));
          r.end_element_ (ns_name);
        }

        template <typename C>
        void XMLCALL event_router<C>::
        characters (void* data, XML_Char const* s, int n)
        {
          event_router& r (*reinterpret_cast<event_router*> (data));
          r.characters_ (s, static_cast<std::size_t> (n));
        }

        template <typename C>
        void event_router<C>::
        start_element_ (XML_Char const* ns_name, XML_Char const** atts)
        {
          if (!buf_.empty ())
          {
            consumer_._characters (buf_);
            buf_.clear ();
          }

          string ns, name, value;
          split_name (ns_name, ns, name);

          consumer_._start_element (ns, name);

          for (; *atts != 0; atts += 2)
          {
            split_name (*atts, ns, name);
            transcoder::assign (value, *(atts + 1));

            consumer_._attribute (ns, name, value);
          }
        }

        template <typename C>
        void event_router<C>::
        end_element_ (XML_Char const* ns_name)
        {
          if (!buf_.empty ())
          {
            consumer_._characters (buf_);
            buf_.clear ();
          }

          string ns, name;
          split_name (ns_name, ns, name);

          consumer_._end_element (ns, name);
        }

        template <typename C>
        void event_router<C>::
        characters_ (XML_Char const* s, std::size_t n)
        {
          if (n != 0)
            transcoder::append (buf_, s, n);
        }

        template <typename C>
        void event_router<C>::
        split_name (XML_Char const* s, string& ns, string& name)
        {
          // Find separator if any.
          //
          size_t n (0);
          for (; s[n] != XML_Char (' ') && s[n] != XML_Char (0); ++n);

          if (s[n] == XML_Char (' '))
          {
            transcoder::assign (ns, s, n);
            transcoder::assign (name, s + n + 1);
          }
          else
          {
            ns.clear ();
            name.assign (s);
          }
        }


        // parser_common
        //


        // file


        template <typename C>
        void parser_common<C>::
        _parse_ (std::basic_string<C> const& file,
                 std::basic_string<C> const& root_namespace,
                 std::basic_string<C> const& root_name)
        {
          std::ifstream ifs;
          ifs.exceptions (std::ios_base::badbit | std::ios_base::failbit);
          ifs.open (file.c_str (), std::ios_base::in | std::ios_base::binary);

          _parse_ (ifs, file, root_namespace, root_name);
        }

        template <typename C>
        void parser_common<C>::
        _parse_ (std::basic_string<C> const& file,
                 std::basic_string<C> const& root_namespace,
                 std::basic_string<C> const& root_name,
                 xml::error_handler<C>& eh)
        {
          std::ifstream ifs;
          ifs.exceptions (std::ios_base::badbit | std::ios_base::failbit);
          ifs.open (file.c_str (), std::ios_base::in | std::ios_base::binary);

          _parse_ (ifs, file, root_namespace, root_name, eh);
        }


        // istream


        template <typename C>
        void parser_common<C>::
        _parse_ (std::istream& is,
                 std::basic_string<C> const& root_namespace,
                 std::basic_string<C> const& root_name)
        {
          error_handler<C> eh;
          _parse_ (is, 0, 0, root_namespace, root_name, eh);
          eh.throw_if_failed ();
        }

        template <typename C>
        void parser_common<C>::
        _parse_ (std::istream& is,
                 std::basic_string<C> const& root_namespace,
                 std::basic_string<C> const& root_name,
                 xml::error_handler<C>& eh)
        {
          if (!_parse_ (is, 0, 0, root_namespace, root_name, eh))
            throw parsing<C> ();
        }

        template <typename C>
        void parser_common<C>::
        _parse_ (std::istream& is,
                 std::basic_string<C> const& system_id,
                 std::basic_string<C> const& root_namespace,
                 std::basic_string<C> const& root_name)
        {
          error_handler<C> eh;
          _parse_ (is, &system_id, 0, root_namespace, root_name, eh);
          eh.throw_if_failed ();
        }

        template <typename C>
        void parser_common<C>::
        _parse_ (std::istream& is,
                 std::basic_string<C> const& system_id,
                 std::basic_string<C> const& root_namespace,
                 std::basic_string<C> const& root_name,
                 xml::error_handler<C>& eh)
        {
          if (!_parse_ (is, &system_id, 0, root_namespace, root_name, eh))
            throw parsing<C> ();
        }

        template <typename C>
        void parser_common<C>::
        _parse_ (std::istream& is,
                 std::basic_string<C> const& system_id,
                 std::basic_string<C> const& public_id,
                 std::basic_string<C> const& root_namespace,
                 std::basic_string<C> const& root_name)
        {
          error_handler<C> eh;
          _parse_ (is, &system_id, &public_id, root_namespace, root_name, eh);
          eh.throw_if_failed ();
        }

        template <typename C>
        void parser_common<C>::
        _parse_ (std::istream& is,
                 std::basic_string<C> const& system_id,
                 std::basic_string<C> const& public_id,
                 std::basic_string<C> const& root_namespace,
                 std::basic_string<C> const& root_name,
                 xml::error_handler<C>& eh)
        {
          if (!_parse_ (
                is, &system_id, &public_id, root_namespace, root_name, eh))
            throw parsing<C> ();
        }

        namespace bits
        {
          struct stream_exception_controller
          {
            ~stream_exception_controller ()
            {
              is_.clear (is_.rdstate () & ~clear_state_);
              is_.exceptions (old_state_);
            }

            stream_exception_controller (std::istream& is,
                                         std::ios_base::iostate except_state,
                                         std::ios_base::iostate clear_state)
                : is_ (is),
                  old_state_ (is_.exceptions ()),
                  clear_state_ (clear_state)
            {
              is_.exceptions (except_state);
            }

          private:
            std::istream& is_;
            std::ios_base::iostate old_state_;
            std::ios_base::iostate clear_state_;
          };
        };

        template <typename C>
        bool parser_common<C>::
        _parse_ (std::istream& is,
                 std::basic_string<C> const* system_id,
                 std::basic_string<C> const* public_id,
                 std::basic_string<C> const& root_namespace,
                 std::basic_string<C> const& root_name,
                 xml::error_handler<C>& eh)
        {
          parser_auto_ptr parser (XML_ParserCreateNS (0, XML_Char (' ')));

          // @@ TODO: Set base.
          //

          context<C> ctx (_parse_begin_ (parser, root_namespace, root_name));

          // Temporarily unset exception failbit. Also clear failbit
          // when we reset the old state.
          //
          bits::stream_exception_controller sec (
            is,
            is.exceptions () & ~std::ios_base::failbit,
            std::ios_base::failbit);

          const std::size_t buf_size = 10240;
          char buf[buf_size];

          do
          {
            is.read (buf, sizeof (buf));

            if (XML_Parse (
                  parser, buf, is.gcount (), is.eof ()) == XML_STATUS_ERROR)
            {
              eh.handle (
                public_id
                ? *public_id
                : (system_id ? *system_id : std::basic_string<C> ()),
                static_cast<unsigned long> (
                  XML_GetCurrentLineNumber (parser)),
                static_cast<unsigned long> (
                  XML_GetCurrentColumnNumber (parser)),
                xml::error_handler<C>::severity::fatal,
                transcoder<XML_LChar, C>::transcode (
                  XML_ErrorString (XML_GetErrorCode (parser))));

              return false;
            }
          } while (!is.eof ());

          _parse_end_ (ctx);

          return true;
        }


        // XML_Parser


        template <typename C>
        context<C> parser_common<C>::
        _parse_begin_ (XML_Parser parser,
                       std::basic_string<C> const& root_namespace,
                       std::basic_string<C> const& root_name)
        {
          return context<C> (parser, *this, root_namespace, root_name);
        }

        template <typename C>
        void parser_common<C>::
        _parse_end_ (context<C>& c)
        {
          c.clear ();
        }


        // parser
        //


        // file


        template <typename X, typename C>
        X parser<X, C>::
        _parse (std::basic_string<C> const& file,
                std::basic_string<C> const& root_namespace,
                std::basic_string<C> const& root_name)
        {
          _parse_ (file, root_namespace, root_name);
          return post ();
        }

        template <typename X, typename C>
        X parser<X, C>::
        _parse (std::basic_string<C> const& file,
                std::basic_string<C> const& root_namespace,
                std::basic_string<C> const& root_name,
                xml::error_handler<C>& eh)
        {
          _parse_ (file, root_namespace, root_name, eh);
          return post ();
        }


        // istream


        template <typename X, typename C>
        X parser<X, C>::
        _parse (std::istream& is,
                std::basic_string<C> const& root_namespace,
                std::basic_string<C> const& root_name)
        {
          _parse_ (is, root_namespace, root_name);
          return post ();
        }

        template <typename X, typename C>
        X parser<X, C>::
        _parse (std::istream& is,
                std::basic_string<C> const& root_namespace,
                std::basic_string<C> const& root_name,
                xml::error_handler<C>& eh)
        {
          _parse_ (is, root_namespace, root_name, eh);
          return post ();
        }

        template <typename X, typename C>
        X parser<X, C>::
        _parse (std::istream& is,
                std::basic_string<C> const& system_id,
                std::basic_string<C> const& root_namespace,
                std::basic_string<C> const& root_name)
        {
          _parse_ (is, system_id, root_namespace, root_name);
          return post ();
        }

        template <typename X, typename C>
        X parser<X, C>::
        _parse (std::istream& is,
                std::basic_string<C> const& system_id,
                std::basic_string<C> const& root_namespace,
                std::basic_string<C> const& root_name,
                xml::error_handler<C>& eh)
        {
          _parse_ (is, system_id, root_namespace, root_name, eh);
          return post ();
        }

        template <typename X, typename C>
        X parser<X, C>::
        _parse (std::istream& is,
                std::basic_string<C> const& system_id,
                std::basic_string<C> const& public_id,
                std::basic_string<C> const& root_namespace,
                std::basic_string<C> const& root_name)
        {
          _parse_ (is, system_id, public_id, root_namespace, root_name);
          return post ();
        }

        template <typename X, typename C>
        X parser<X, C>::
        _parse (std::istream& is,
                std::basic_string<C> const& system_id,
                std::basic_string<C> const& public_id,
                std::basic_string<C> const& root_namespace,
                std::basic_string<C> const& root_name,
                xml::error_handler<C>& eh)
        {
          _parse_ (is, system_id, public_id, root_namespace, root_name, eh);
          return post ();
        }


        // XML_parser


        template <typename X, typename C>
        context<C> parser<X, C>::
        _parse_begin (XML_Parser parser,
                      std::basic_string<C> const& root_namespace,
                      std::basic_string<C> const& root_name)
        {
          return _parse_begin_ (parser, root_namespace, root_name);
        }

        template <typename X, typename C>
        X parser<X, C>::
        _parse_end (context<C>& c)
        {
          XML_Error e (XML_GetErrorCode (c.parser_));

          _parse_end_ (c);

          if (e != XML_ERROR_NONE)
            throw parsing<C> ();

          return post ();
        }


        // parser<void>
        //


        // file


        template <typename C>
        void parser<void, C>::
        _parse (std::basic_string<C> const& file,
                std::basic_string<C> const& root_namespace,
                std::basic_string<C> const& root_name)
        {
          _parse_ (file, root_namespace, root_name);
          post ();
        }

        template <typename C>
        void parser<void, C>::
        _parse (std::basic_string<C> const& file,
                std::basic_string<C> const& root_namespace,
                std::basic_string<C> const& root_name,
                xml::error_handler<C>& eh)
        {
          _parse_ (file, root_namespace, root_name, eh);
          post ();
        }


        // istream


        template <typename C>
        void parser<void, C>::
        _parse (std::istream& is,
                std::basic_string<C> const& root_namespace,
                std::basic_string<C> const& root_name)
        {
          _parse_ (is, root_namespace, root_name);
          post ();
        }

        template <typename C>
        void parser<void, C>::
        _parse (std::istream& is,
                std::basic_string<C> const& root_namespace,
                std::basic_string<C> const& root_name,
                xml::error_handler<C>& eh)
        {
          _parse_ (is, root_namespace, root_name, eh);
          post ();
        }

        template <typename C>
        void parser<void, C>::
        _parse (std::istream& is,
                std::basic_string<C> const& system_id,
                std::basic_string<C> const& root_namespace,
                std::basic_string<C> const& root_name)
        {
          _parse_ (is, system_id, root_namespace, root_name);
          post ();
        }

        template <typename C>
        void parser<void, C>::
        _parse (std::istream& is,
                std::basic_string<C> const& system_id,
                std::basic_string<C> const& root_namespace,
                std::basic_string<C> const& root_name,
                xml::error_handler<C>& eh)
        {
          _parse_ (is, system_id, root_namespace, root_name, eh);
          post ();
        }

        template <typename C>
        void parser<void, C>::
        _parse (std::istream& is,
                std::basic_string<C> const& system_id,
                std::basic_string<C> const& public_id,
                std::basic_string<C> const& root_namespace,
                std::basic_string<C> const& root_name)
        {
          _parse_ (is, system_id, public_id, root_namespace, root_name);
          post ();
        }

        template <typename C>
        void parser<void, C>::
        _parse (std::istream& is,
                std::basic_string<C> const& system_id,
                std::basic_string<C> const& public_id,
                std::basic_string<C> const& root_namespace,
                std::basic_string<C> const& root_name,
                xml::error_handler<C>& eh)
        {
          _parse_ (is, system_id, public_id, root_namespace, root_name, eh);
          post ();
        }


        // XML_parser


        template <typename C>
        context<C> parser<void, C>::
        _parse_begin (XML_Parser parser,
                      std::basic_string<C> const& root_namespace,
                      std::basic_string<C> const& root_name)
        {
          return _parse_begin_ (parser, root_namespace, root_name);
        }

        template <typename C>
        void parser<void, C>::
        _parse_end (context<C>& c)
        {
          XML_Error e (XML_GetErrorCode (c.parser_));

          _parse_end_ (c);

          if (e != XML_ERROR_NONE)
            throw parsing<C> ();

          post ();
        }
      }
    }
  }
}
