// file      : xsd/cxx/parser/elements.txx
// author    : Boris Kolpackov <boris@codesynthesis.com>
// copyright : Copyright (c) 2005-2006 Code Synthesis Tools CC
// license   : GNU GPL v2 + exceptions; see accompanying LICENSE file

#include <cassert>

#include <xsd/cxx/parser/exceptions.hxx>

namespace xsd
{
  namespace cxx
  {
    namespace parser
    {
      // event_consumer
      //
      template <typename C>
      event_consumer<C>::
      ~event_consumer ()
      {
      }

      template <typename C>
      void event_consumer<C>::
      _start_element (std::basic_string<C> const& ns,
                      std::basic_string<C> const& n)
      {
        _start_unexpected_element (ns, n);
      }

      template <typename C>
      void event_consumer<C>::
      _end_element (std::basic_string<C> const& ns,
                    std::basic_string<C> const& n)
      {
        _end_unexpected_element (ns, n);
      }

      template <typename C>
      void event_consumer<C>::
      _attribute (std::basic_string<C> const& ns,
                  std::basic_string<C> const& n,
                  std::basic_string<C> const& v)
      {
        _unexpected_attribute (ns, n, v);
      }

      template <typename C>
      void event_consumer<C>::
      _characters (std::basic_string<C> const& s)
      {
        _unexpected_characters (s);
      }

      template <typename C>
      void event_consumer<C>::
      _start_unexpected_element (std::basic_string<C> const&,
                                 std::basic_string<C> const&)
      {
      }

      template <typename C>
      void event_consumer<C>::
      _end_unexpected_element (std::basic_string<C> const&,
                               std::basic_string<C> const&)
      {
      }

      template <typename C>
      void event_consumer<C>::
      _unexpected_attribute (std::basic_string<C> const&,
                             std::basic_string<C> const&,
                             std::basic_string<C> const&)
      {
      }

      template <typename C>
      void event_consumer<C>::
      _unexpected_characters (std::basic_string<C> const&)
      {
      }


      // event_consumer_impl
      //

      template <typename C>
      void event_consumer_impl<C>::
      _start_element (std::basic_string<C> const& ns,
                      std::basic_string<C> const& name)
      {
        state& s (context_.top ());

        if (s.depth_++ > 0)
        {
          if (s.parser_)
            s.parser_->_start_element (ns, name);
        }
        else
        {
          if (_start_element_impl (ns, name))
          {
            if (s.parser_)
              s.parser_->context_.push (state ());
          }
          else
            _start_unexpected_element (ns, name);
        }
      }

      template <typename C>
      void event_consumer_impl<C>::
      _end_element (std::basic_string<C> const& ns,
                    std::basic_string<C> const& name)
      {
        if (context_.top ().depth_ == 0)
          context_.pop ();

        state& s (context_.top ());

        if (--s.depth_ > 0)
        {
          if (s.parser_)
            s.parser_->_end_element (ns, name);
        }
        else
        {
          if (!_end_element_impl (ns, name))
            _end_unexpected_element (ns, name);
        }
      }

      template <typename C>
      void event_consumer_impl<C>::
      _attribute (std::basic_string<C> const& ns,
                  std::basic_string<C> const& name,
                  std::basic_string<C> const& value)
      {
        state& s (context_.top ());

        if (s.depth_ > 0)
        {
          if (s.parser_)
            s.parser_->_attribute (ns, name, value);
        }
        else
        {
          if (!_attribute_impl (ns, name, value))
            _unexpected_attribute (ns, name, value);
        }
      }

      template <typename C>
      void event_consumer_impl<C>::
      _characters (std::basic_string<C> const& str)
      {
        state& s (context_.top ());

        if (s.depth_ > 0)
        {
          if (s.parser_)
            s.parser_->_characters (str);
        }
        else
        {
          if (!_characters_impl (str))
            _unexpected_characters (str);
        }
      }


      template <typename C>
      bool event_consumer_impl<C>::
      _start_element_impl (std::basic_string<C> const&,
                           std::basic_string<C> const&)
      {
        return false;
      }

      template <typename C>
      bool event_consumer_impl<C>::
      _end_element_impl (std::basic_string<C> const&,
                         std::basic_string<C> const&)
      {
        return false;
      }

      template <typename C>
      bool event_consumer_impl<C>::
      _attribute_impl (std::basic_string<C> const&,
                       std::basic_string<C> const&,
                       std::basic_string<C> const&)
      {
        return false;
      }

      template <typename C>
      bool event_consumer_impl<C>::
      _characters_impl (std::basic_string<C> const&)
      {
        return false;
      }

      // document
      //
      template <typename C>
      document<C>::
      document (event_consumer_impl<C>& root,
                string const& ns,
                string const& name)
          : root_ (root), name_ (name), ns_ (ns), depth_ (0)
      {
      }

      template <typename C>
      void document<C>::
      _start_element (string const& ns, string const& name)
      {
        if (depth_++ > 0)
        {
          root_._start_element (ns, name);
        }
        else
        {
          if (name_ == name && ns_ == ns)
          {
            root_.pre ();
            root_.context_.push (
              typename event_consumer_impl<C>::state ());
          }
          else
            throw unexpected_element<C> (name, ns, name_, ns_);
        }
      }

      template <typename C>
      void document<C>::
      _end_element (string const& ns, string const& name)
      {
        assert (depth_ > 0);

        if (--depth_ > 0)
          root_._end_element (ns, name);
        else
        {
          if (name_ == name && ns_ == ns)
          {
            root_.context_.pop ();
            //
            // post() will be called by one of the _parse() functions.
          }
          else
            throw unexpected_element<C> (name, ns, name_, ns_);
        }
      }

      template <typename C>
      void document<C>::
      _attribute (string const& ns,
                  string const& name,
                  string const& value)
      {
        assert (depth_ > 0);
        root_._attribute (ns, name, value);
      }

      template <typename C>
      void document<C>::
      _characters (std::basic_string<C> const& s)
      {
        assert (depth_ > 0);
        root_._characters (s);
      }


      // list
      //
      template <typename I, typename C, template<typename, typename> class T>
      void list<I, C, T>::
      _characters (std::basic_string<C> const& s)
      {
        if (item_ == 0)
          return;

        //@@ In case of huge lists, it would be more efficient
        //   not to use "total" consolidation. Instead consolidate
        //   here until the next space.
        //

        // Note that according to the spec the separator is exactly
        // one space (0x20). This makes our life much easier.
        //

        for (std::size_t i (0), j (s.find (C (' ')));;)
        {
          if (j != std::basic_string<C>::npos)
          {
            item_->pre ();
            item_->_characters (std::basic_string<C> (s, i, j - i));
            item (item_->post ());

            i = j + 1;
            j = s.find (C (' '), i);
          }
          else
          {
            // Last element.
            //
            item_->pre ();
            item_->_characters (std::basic_string<C> (s, i));
            item (item_->post ());

            break;
          }
        }
      }

      template <typename C, template<typename, typename> class T>
      void list<void, C, T>::
      _characters (std::basic_string<C> const& s)
      {
        if (item_ == 0)
          return;

        //@@ In case of huge lists, it would be more efficient
        //   not to use "total" consolidation. Instead consolidate
        //   here until the next space.
        //

        // Note that according to the spec the separator is exactly
        // one space (0x20). This makes our life much easier.
        //

        for (std::size_t i (0), j (s.find (C (' ')));;)
        {
          if (j != std::basic_string<C>::npos)
          {
            item_->pre ();
            item_->_characters (std::basic_string<C> (s, i, j - i));
            item_->post ();
            item ();

            i = j + 1;
            j = s.find (C (' '), i);
          }
          else
          {
            // Last element.
            //
            item_->pre ();
            item_->_characters (std::basic_string<C> (s, i));
            item_->post ();
            item ();

            break;
          }
        }
      }
    }
  }
}
