// file      : xsd/cxx/parser/elements.hxx
// author    : Boris Kolpackov <boris@codesynthesis.com>
// copyright : Copyright (c) 2005-2006 Code Synthesis Tools CC
// license   : GNU GPL v2 + exceptions; see accompanying LICENSE file

#ifndef XSD_CXX_PARSER_ELEMENTS_HXX
#define XSD_CXX_PARSER_ELEMENTS_HXX

#include <stack>
#include <string>
#include <cstdlib> // std::size_t

namespace xsd
{
  namespace cxx
  {
    namespace parser
    {
      //
      //
      template <typename C>
      struct event_consumer
      {
        virtual
        ~event_consumer ();

        virtual void
        _start_element (std::basic_string<C> const&,  // namespace
                        std::basic_string<C> const&); // name

        virtual void
        _end_element (std::basic_string<C> const&,  // namespace
                      std::basic_string<C> const&); // name

        virtual void
        _attribute (std::basic_string<C> const&,  // namespace
                    std::basic_string<C> const&,  // name
                    std::basic_string<C> const&); // value

        virtual void
        _characters (std::basic_string<C> const&);


        // These functions are called when "unexpected" content
        // is encountered. Use them to handle "type-less" content
        // such as mixed content models, any/anyAttribute, and
        // anyType/anySimpleType. By default these functions do
        // nothing.
        //
        virtual void
        _start_unexpected_element (std::basic_string<C> const&,  // namespace
                                   std::basic_string<C> const&); // name

        virtual void
        _end_unexpected_element (std::basic_string<C> const&,  // namespace
                                 std::basic_string<C> const&); // name

        virtual void
        _unexpected_attribute (std::basic_string<C> const&,  // namespace
                               std::basic_string<C> const&,  // name
                               std::basic_string<C> const&); // value

        virtual void
        _unexpected_characters (std::basic_string<C> const&);
      };


      //
      //
      template <typename C>
      struct event_consumer_impl: event_consumer<C>
      {
        virtual void
        pre ()
        {
        }

        //
        //
        virtual void
        _start_element (std::basic_string<C> const& ns,
                        std::basic_string<C> const& name);

        virtual void
        _end_element (std::basic_string<C> const& ns,
                      std::basic_string<C> const& name);

        virtual void
        _attribute (std::basic_string<C> const& ns,
                    std::basic_string<C> const& name,
                    std::basic_string<C> const& value);

        virtual void
        _characters (std::basic_string<C> const&);


        //
        //
        virtual bool
        _start_element_impl (std::basic_string<C> const& ns,
                             std::basic_string<C> const& name);

        virtual bool
        _end_element_impl (std::basic_string<C> const& ns,
                           std::basic_string<C> const& name);

        virtual bool
        _attribute_impl (std::basic_string<C> const& ns,
                         std::basic_string<C> const& name,
                         std::basic_string<C> const& value);

        virtual bool
        _characters_impl (std::basic_string<C> const&);


      protected:
        struct state
        {
          state ()
              : depth_ (0), parser_ (0)
          {
          }

          std::size_t depth_;
          event_consumer_impl<C>* parser_;
        };

        std::stack<state> context_;

        template <typename>
        friend struct document;
      };


      //
      //
      template <typename C>
      struct document: event_consumer<C>
      {
        typedef std::basic_string<C> string;

        document (event_consumer_impl<C>& root,
                  string const& ns,
                  string const& name);

        virtual void
        _start_element (string const& ns, string const& name);

        virtual void
        _end_element (string const& ns, string const& name);

        virtual void
        _attribute (string const& ns,
                    string const& name,
                    string const& value);

        virtual void
        _characters (std::basic_string<C> const&);

      private:
        event_consumer_impl<C>& root_;
        string name_;
        string ns_;
        std::size_t depth_;
      };


      // Template for xsd:list
      //
      template <typename I, typename C, template<typename, typename> class T>
      struct list: virtual event_consumer_impl<C>
      {
        list ()
            : item_ (0)
        {
        }

        // Parser hooks. Override them in your implementation.
        //
        virtual void
        item (I const&)
        {
        }

        // Parser construction API.
        //
        void
        item_parser (T<I, C>& item)
        {
          item_ = &item;
        }

        void
        parsers (T<I, C>& item)
        {
          item_ = &item;
        }

        // Implementation.
        //

        virtual void
        _characters (std::basic_string<C> const& s);

      private:
        T<I, C>* item_;
      };


      // Specialization for void.
      //
      template <typename C, template<typename, typename> class T>
      struct list<void, C, T>: virtual event_consumer_impl<C>
      {
        list ()
            : item_ (0)
        {
        }

        // Parser hooks. Override them in your implementation.
        //
        virtual void
        item ()
        {
        }

        // Parser construction API.
        //
        void
        item_parser (T<void, C>& item)
        {
          item_ = &item;
        }

        void
        parsers (T<void, C>& item)
        {
          item_ = &item;
        }

        // Implementation.
        //

        virtual void
        _characters (std::basic_string<C> const& s);

      private:
        T<void, C>* item_;
      };


      // Hook call glue.
      //
      template <typename T>
      struct glue
      {
        template <typename X, typename Y>
        static void
        call (X* x, void (X::*hook) (T const&),
              Y* y, T (Y::*post) ())
        {
          (x->*hook) ((y->*post) ());
        }
      };

      template <>
      struct glue<void>
      {
        template <typename X, typename Y>
        static void
        call (X* x, void (X::*hook) (),
              Y* y, void (Y::*post) ())
        {
          (y->*post) ();
          (x->*hook) ();
        }
      };
    }
  }
}

#include <xsd/cxx/parser/elements.txx>

#endif  // XSD_CXX_PARSER_ELEMENTS_HXX
