// file      : xsd/cxx/parser/xerces/elements.txx
// author    : Boris Kolpackov <boris@codesynthesis.com>
// copyright : Copyright (c) 2005-2006 Code Synthesis Tools CC
// license   : GNU GPL v2 + exceptions; see accompanying LICENSE file

#include <istream>

#include <xercesc/sax2/Attributes.hpp>
#include <xercesc/sax2/XMLReaderFactory.hpp>

#include <xsd/cxx/xml/string.hxx>
#include <xsd/cxx/xml/sax/std-input-source.hxx>
#include <xsd/cxx/xml/sax/bits/error-handler-proxy.hxx>

#include <xsd/cxx/parser/error-handler.hxx>

namespace xsd
{
  namespace cxx
  {
    namespace parser
    {
      namespace xerces
      {
        // event_router
        //

        template <typename C>
        void event_router<C>::
        startElement(XMLCh const* const uri,
                     XMLCh const* const lname,
                     XMLCh const* const /*qname*/,
                     xercesc::Attributes const& attributes)
        {
          if (!buf_.empty ())
          {
            t_._characters (buf_);
            buf_.clear ();
          }

          string ns (xml::transcode<C> (uri));
          string name (xml::transcode<C> (lname));

          t_._start_element (ns, name);

          for (unsigned long i (0); i < attributes.getLength(); ++i)
          {
            string ns (xml::transcode<C> (attributes.getURI (i)));

            string name (xml::transcode<C> (attributes.getLocalName (i)));

            string value (xml::transcode<C> (attributes.getValue (i)));

            t_._attribute (ns, name, value);
          }
        }

        template <typename C>
        void event_router<C>::
        endElement(XMLCh const* const uri,
                   XMLCh const* const lname,
                   XMLCh const* const /*qname*/)
        {
          if (!buf_.empty ())
          {
            t_._characters (buf_);
            buf_.clear ();
          }

          string ns (xml::transcode<C> (uri));
          string name (xml::transcode<C> (lname));

          t_._end_element (ns, name);
        }

        template <typename C>
        void event_router<C>::
        characters (XMLCh const* const s, unsigned int const n)
        {
          if (n != 0)
            buf_ += xml::transcode<C> (s, n);
        }


        // traverser_common
        //

        // parse (uri)
        //
        template <typename C>
        void traverser_common<C>::
        _parse_ (std::basic_string<C> const& uri,
                 std::basic_string<C> const& root_namespace,
                 std::basic_string<C> const& root_name,
                 flags f,
                 properties<C> const& p)
        {
          xml::auto_initializer init ((f & flags::dont_initialize) == 0);

          error_handler<C> eh;
          xml::sax::bits::error_handler_proxy<C> eh_proxy (eh);
          std::auto_ptr<xercesc::SAX2XMLReader> sax (_create_sax (f, p));

          _parse_ (uri, root_namespace, root_name, eh_proxy, *sax, f, p);

          eh.throw_if_failed ();
        }

        // error_handler
        //

        template <typename C>
        void traverser_common<C>::
        _parse_ (std::basic_string<C> const& uri,
                 std::basic_string<C> const& root_namespace,
                 std::basic_string<C> const& root_name,
                 xml::error_handler<C>& eh,
                 flags f,
                 properties<C> const& p)
        {
          xml::auto_initializer init ((f & flags::dont_initialize) == 0);

          xml::sax::bits::error_handler_proxy<C> eh_proxy (eh);
          std::auto_ptr<xercesc::SAX2XMLReader> sax (_create_sax (f, p));

          _parse_ (uri, root_namespace, root_name, eh_proxy, *sax, f, p);

          if (eh_proxy.failed ())
            throw parsing<C> ();
        }

        // ErrorHandler
        //

        template <typename C>
        void traverser_common<C>::
        _parse_ (std::basic_string<C> const& uri,
                 std::basic_string<C> const& root_namespace,
                 std::basic_string<C> const& root_name,
                 xercesc::ErrorHandler& eh,
                 flags f,
                 properties<C> const& p)
        {
          xml::sax::bits::error_handler_proxy<C> eh_proxy (eh);
          std::auto_ptr<xercesc::SAX2XMLReader> sax (_create_sax (f, p));

          _parse_ (uri, root_namespace, root_name, eh_proxy, *sax, f, p);

          if (eh_proxy.failed ())
            throw parsing<C> ();
        }

        // SAX2XMLReader
        //

        template <typename C>
        void traverser_common<C>::
        _parse_ (std::basic_string<C> const& uri,
                 std::basic_string<C> const& root_namespace,
                 std::basic_string<C> const& root_name,
                 xercesc::SAX2XMLReader& sax,
                 flags f,
                 properties<C> const& p)
        {
          // If there is no error handler, then fall back on the default
          // implementation.
          //
          xercesc::ErrorHandler* eh (sax.getErrorHandler ());

          if (eh)
          {
            xml::sax::bits::error_handler_proxy<C> eh_proxy (*eh);

            _parse_ (uri, root_namespace, root_name, eh_proxy, sax, f, p);

            if (eh_proxy.failed ())
              throw parsing<C> ();
          }
          else
          {
            error_handler<C> fallback_eh;
            xml::sax::bits::error_handler_proxy<C> eh_proxy (fallback_eh);

            _parse_ (uri, root_namespace, root_name, eh_proxy, sax, f, p);

            fallback_eh.throw_if_failed ();
          }
        }


        // parse (istream)
        //

        template <typename C>
        void traverser_common<C>::
        _parse_ (std::istream& is,
                 std::basic_string<C> const& root_namespace,
                 std::basic_string<C> const& root_name,
                 flags f,
                 properties<C> const& p)
        {
          xml::auto_initializer init ((f & flags::dont_initialize) == 0);

          xml::sax::std_input_source isrc (is);

          _parse_ (isrc, root_namespace, root_name, f, p);
        }

        template <typename C>
        void traverser_common<C>::
        _parse_ (std::istream& is,
                 std::basic_string<C> const& root_namespace,
                 std::basic_string<C> const& root_name,
                 xml::error_handler<C>& eh,
                 flags f,
                 properties<C> const& p)
        {
          xml::auto_initializer init ((f & flags::dont_initialize) == 0);

          xml::sax::std_input_source isrc (is);

          _parse_ (isrc, root_namespace, root_name, eh, f, p);
        }

        template <typename C>
        void traverser_common<C>::
        _parse_ (std::istream& is,
                 std::basic_string<C> const& root_namespace,
                 std::basic_string<C> const& root_name,
                 xercesc::ErrorHandler& eh,
                 flags f,
                 properties<C> const& p)
        {
          xml::sax::std_input_source isrc (is);

          _parse_ (isrc, root_namespace, root_name, eh, f, p);
        }

        template <typename C>
        void traverser_common<C>::
        _parse_ (std::istream& is,
                 std::basic_string<C> const& root_namespace,
                 std::basic_string<C> const& root_name,
                 xercesc::SAX2XMLReader& sax,
                 flags f,
                 properties<C> const& p)
        {
          xml::sax::std_input_source isrc (is);

          _parse_ (isrc, root_namespace, root_name, sax, f, p);
        }


        // parse (istream, system_id)
        //


        template <typename C>
        void traverser_common<C>::
        _parse_ (std::istream& is,
                 std::basic_string<C> const& system_id,
                 std::basic_string<C> const& root_namespace,
                 std::basic_string<C> const& root_name,
                 flags f,
                 properties<C> const& p)
        {
          xml::auto_initializer init ((f & flags::dont_initialize) == 0);

          xml::sax::std_input_source isrc (is, system_id);

          _parse_ (isrc, root_namespace, root_name, f, p);
        }

        template <typename C>
        void traverser_common<C>::
        _parse_ (std::istream& is,
                 std::basic_string<C> const& system_id,
                 std::basic_string<C> const& root_namespace,
                 std::basic_string<C> const& root_name,
                 xml::error_handler<C>& eh,
                 flags f,
                 properties<C> const& p)
        {
          xml::auto_initializer init ((f & flags::dont_initialize) == 0);

          xml::sax::std_input_source isrc (is, system_id);

          _parse_ (isrc, root_namespace, root_name, eh, f, p);
        }

        template <typename C>
        void traverser_common<C>::
        _parse_ (std::istream& is,
                 std::basic_string<C> const& system_id,
                 std::basic_string<C> const& root_namespace,
                 std::basic_string<C> const& root_name,
                 xercesc::ErrorHandler& eh,
                 flags f,
                 properties<C> const& p)
        {
          xml::sax::std_input_source isrc (is, system_id);

          _parse_ (isrc, root_namespace, root_name, eh, f, p);
        }

        template <typename C>
        void traverser_common<C>::
        _parse_ (std::istream& is,
                 std::basic_string<C> const& system_id,
                 std::basic_string<C> const& root_namespace,
                 std::basic_string<C> const& root_name,
                 xercesc::SAX2XMLReader& sax,
                 flags f,
                 properties<C> const& p)
        {
          xml::sax::std_input_source isrc (is, system_id);

          _parse_ (isrc, root_namespace, root_name, sax, f, p);
        }


        // parse (istream, system_id, public_id)
        //

        template <typename C>
        void traverser_common<C>::
        _parse_ (std::istream& is,
                 std::basic_string<C> const& system_id,
                 std::basic_string<C> const& public_id,
                 std::basic_string<C> const& root_namespace,
                 std::basic_string<C> const& root_name,
                 flags f,
                 properties<C> const& p)
        {
          xml::auto_initializer init ((f & flags::dont_initialize) == 0);

          xml::sax::std_input_source isrc (is, system_id, public_id);

          _parse_ (isrc, root_namespace, root_name, f, p);
        }

        template <typename C>
        void traverser_common<C>::
        _parse_ (std::istream& is,
                 std::basic_string<C> const& system_id,
                 std::basic_string<C> const& public_id,
                 std::basic_string<C> const& root_namespace,
                 std::basic_string<C> const& root_name,
                 xml::error_handler<C>& eh,
                 flags f,
                 properties<C> const& p)
        {
          xml::auto_initializer init ((f & flags::dont_initialize) == 0);

          xml::sax::std_input_source isrc (is, system_id, public_id);

          _parse_ (isrc, root_namespace, root_name, eh, f, p);
        }

        template <typename C>
        void traverser_common<C>::
        _parse_ (std::istream& is,
                 std::basic_string<C> const& system_id,
                 std::basic_string<C> const& public_id,
                 std::basic_string<C> const& root_namespace,
                 std::basic_string<C> const& root_name,
                 xercesc::ErrorHandler& eh,
                 flags f,
                 properties<C> const& p)
        {
          xml::sax::std_input_source isrc (is, system_id, public_id);

          _parse_ (isrc, root_namespace, root_name, eh, f, p);
        }

        template <typename C>
        void traverser_common<C>::
        _parse_ (std::istream& is,
                 std::basic_string<C> const& system_id,
                 std::basic_string<C> const& public_id,
                 std::basic_string<C> const& root_namespace,
                 std::basic_string<C> const& root_name,
                 xercesc::SAX2XMLReader& sax,
                 flags f,
                 properties<C> const& p)
        {
          xml::sax::std_input_source isrc (is, system_id, public_id);

          _parse_ (isrc, root_namespace, root_name, sax, f, p);
        }


        // parse (InputSource)
        //


        template <typename C>
        void traverser_common<C>::
        _parse_ (xercesc::InputSource const& is,
                 std::basic_string<C> const& root_namespace,
                 std::basic_string<C> const& root_name,
                 flags f,
                 properties<C> const& p)
        {
          error_handler<C> eh;
          xml::sax::bits::error_handler_proxy<C> eh_proxy (eh);
          std::auto_ptr<xercesc::SAX2XMLReader> sax (_create_sax (f, p));

          _parse_ (is, root_namespace, root_name, eh_proxy, *sax, f, p);

          eh.throw_if_failed ();
        }

        template <typename C>
        void traverser_common<C>::
        _parse_ (xercesc::InputSource const& is,
                 std::basic_string<C> const& root_namespace,
                 std::basic_string<C> const& root_name,
                 xml::error_handler<C>& eh,
                 flags f,
                 properties<C> const& p)
        {
          xml::sax::bits::error_handler_proxy<C> eh_proxy (eh);
          std::auto_ptr<xercesc::SAX2XMLReader> sax (_create_sax (f, p));

          _parse_ (is, root_namespace, root_name, eh_proxy, *sax, f, p);

          if (eh_proxy.failed ())
            throw parsing<C> ();
        }

        template <typename C>
        void traverser_common<C>::
        _parse_ (xercesc::InputSource const& is,
                 std::basic_string<C> const& root_namespace,
                 std::basic_string<C> const& root_name,
                 xercesc::ErrorHandler& eh,
                 flags f,
                 properties<C> const& p)
        {
          xml::sax::bits::error_handler_proxy<C> eh_proxy (eh);
          std::auto_ptr<xercesc::SAX2XMLReader> sax (_create_sax (f, p));

          _parse_ (is, root_namespace, root_name, eh_proxy, *sax, f, p);

          if (eh_proxy.failed ())
            throw parsing<C> ();
        }


        template <typename C>
        void traverser_common<C>::
        _parse_ (xercesc::InputSource const& is,
                 std::basic_string<C> const& root_namespace,
                 std::basic_string<C> const& root_name,
                 xercesc::SAX2XMLReader& sax,
                 flags f,
                 properties<C> const& p)
        {
          // If there is no error handler, then fall back on the default
          // implementation.
          //
          xercesc::ErrorHandler* eh (sax.getErrorHandler ());

          if (eh)
          {
            xml::sax::bits::error_handler_proxy<C> eh_proxy (*eh);

            _parse_ (is, root_namespace, root_name, eh_proxy, sax, f, p);

            if (eh_proxy.failed ())
              throw parsing<C> ();
          }
          else
          {
            error_handler<C> fallback_eh;
            xml::sax::bits::error_handler_proxy<C> eh_proxy (fallback_eh);

            _parse_ (is, root_namespace, root_name, eh_proxy, sax, f, p);

            fallback_eh.throw_if_failed ();
          }
        }

        namespace Bits
        {
          struct ErrorHandlingController
          {
            ErrorHandlingController (xercesc::SAX2XMLReader& sax,
                                     xercesc::ErrorHandler& eh)
                : sax_ (sax), eh_ (sax_.getErrorHandler ())
            {
              sax_.setErrorHandler (&eh);
            }

            ~ErrorHandlingController ()
            {
              sax_.setErrorHandler (eh_);
            }

          private:
            xercesc::SAX2XMLReader& sax_;
            xercesc::ErrorHandler* eh_;
          };

          struct ContentHandlingController
          {
            ContentHandlingController (xercesc::SAX2XMLReader& sax,
                                       xercesc::ContentHandler& ch)
                : sax_ (sax), ch_ (sax_.getContentHandler ())
            {
              sax_.setContentHandler (&ch);
            }

            ~ContentHandlingController ()
            {
              sax_.setContentHandler (ch_);
            }

          private:
            xercesc::SAX2XMLReader& sax_;
            xercesc::ContentHandler* ch_;
          };
        };

        template <typename C>
        void traverser_common<C>::
        _parse_ (std::basic_string<C> const& uri,
                 std::basic_string<C> const& root_namespace,
                 std::basic_string<C> const& root_name,
                 xercesc::ErrorHandler& eh,
                 xercesc::SAX2XMLReader& sax,
                 flags,
                 properties<C> const&)
        {
          document<C> doc (*this, root_namespace, root_name);
          event_router<C> router (doc);

          Bits::ErrorHandlingController ehc (sax, eh);
          Bits::ContentHandlingController chc (sax, router);

          sax.parse (xml::string (uri).c_str ());
        }

        template <typename C>
        void traverser_common<C>::
        _parse_ (xercesc::InputSource const& is,
                 std::basic_string<C> const& root_namespace,
                 std::basic_string<C> const& root_name,
                 xercesc::ErrorHandler& eh,
                 xercesc::SAX2XMLReader& sax,
                 flags,
                 properties<C> const&)
        {
          document<C> doc (*this, root_namespace, root_name);
          event_router<C> router (doc);

          Bits::ErrorHandlingController controller (sax, eh);
          Bits::ContentHandlingController chc (sax, router);

          sax.parse (is);
        }


        template <typename C>
        std::auto_ptr<xercesc::SAX2XMLReader> traverser_common<C>::
        _create_sax (flags f, properties<C> const& p)
        {
          // HP aCC cannot handle using namespace xercesc;
          //
          using xercesc::SAX2XMLReader;
          using xercesc::XMLReaderFactory;
          using xercesc::XMLUni;

          std::auto_ptr<SAX2XMLReader> sax (
            XMLReaderFactory::createXMLReader());

          sax->setFeature(XMLUni::fgSAX2CoreNameSpaces, true);
          sax->setFeature(XMLUni::fgSAX2CoreNameSpacePrefixes, true);
          sax->setFeature (XMLUni::fgXercesValidationErrorAsFatal, true);

          if ((f & flags::dont_validate) == 0)
          {
            sax->setFeature(XMLUni::fgSAX2CoreValidation, true);
            sax->setFeature(XMLUni::fgXercesSchema, true);
            sax->setFeature(XMLUni::fgXercesSchemaFullChecking, true);
          }
          else
          {
            sax->setFeature(XMLUni::fgSAX2CoreValidation, false);
            sax->setFeature(XMLUni::fgXercesSchema, false);
            sax->setFeature(XMLUni::fgXercesSchemaFullChecking, false);
          }

          // Transfer properies if any.
          //

          if (!p.schema_location ().empty ())
          {
            xml::string sl (p.schema_location ());
            void const* v (sl.c_str ());

            sax->setProperty (
              XMLUni::fgXercesSchemaExternalSchemaLocation,
              const_cast<void*> (v));
          }

          if (!p.no_namespace_schema_location ().empty ())
          {
            xml::string sl (p.no_namespace_schema_location ());
            void const* v (sl.c_str ());

            sax->setProperty (
              XMLUni::fgXercesSchemaExternalNoNameSpaceSchemaLocation,
              const_cast<void*> (v));
          }

          return sax;
        }


        // traverser
        //


        // parse (uri)
        //
        template <typename X, typename C>
        X traverser<X, C>::
        _parse (std::basic_string<C> const& uri,
                std::basic_string<C> const& root_namespace,
                std::basic_string<C> const& root_name,
                flags f,
                properties<C> const& p)
        {
          _parse_ (uri, root_namespace, root_name, f, p);
          return post ();
        }


        template <typename X, typename C>
        X traverser<X, C>::
        _parse (C const* uri,
                std::basic_string<C> const& root_namespace,
                std::basic_string<C> const& root_name,
                flags f,
                properties<C> const& p)
        {
          _parse_ (
            std::basic_string<C> (uri), root_namespace, root_name, f, p);
          return post ();
        }

        // error_handler
        //

        template <typename X, typename C>
        X traverser<X, C>::
        _parse (std::basic_string<C> const& uri,
                std::basic_string<C> const& root_namespace,
                std::basic_string<C> const& root_name,
                xml::error_handler<C>& eh,
                flags f,
                properties<C> const& p)
        {
          _parse_ (uri, root_namespace, root_name, eh, f, p);
          return post ();
        }


        template <typename X, typename C>
        X traverser<X, C>::
        _parse (C const* uri,
                std::basic_string<C> const& root_namespace,
                std::basic_string<C> const& root_name,
                xml::error_handler<C>& eh,
                flags f,
                properties<C> const& p)
        {
          _parse_ (
            std::basic_string<C> (uri), root_namespace, root_name, eh, f, p);

          return post ();
        }

        // ErrorHandler
        //

        template <typename X, typename C>
        X traverser<X, C>::
        _parse (std::basic_string<C> const& uri,
                std::basic_string<C> const& root_namespace,
                std::basic_string<C> const& root_name,
                xercesc::ErrorHandler& eh,
                flags f,
                properties<C> const& p)
        {
          _parse_ (uri, root_namespace, root_name, eh, f, p);
          return post ();
        }


        template <typename X, typename C>
        X traverser<X, C>::
        _parse (C const* uri,
                std::basic_string<C> const& root_namespace,
                std::basic_string<C> const& root_name,
                xercesc::ErrorHandler& eh,
                flags f,
                properties<C> const& p)
        {
          _parse_ (
            std::basic_string<C> (uri), root_namespace, root_name, eh, f, p);
          return post ();
        }

        // SAX2XMLReader
        //

        template <typename X, typename C>
        X traverser<X, C>::
        _parse (std::basic_string<C> const& uri,
                std::basic_string<C> const& root_namespace,
                std::basic_string<C> const& root_name,
                xercesc::SAX2XMLReader& sax,
                flags f,
                properties<C> const& p)
        {
          _parse_ (uri, root_namespace, root_name, sax, f, p);
          return post ();
        }

        template <typename X, typename C>
        X traverser<X, C>::
        _parse (C const* uri,
                std::basic_string<C> const& root_namespace,
                std::basic_string<C> const& root_name,
                xercesc::SAX2XMLReader& sax,
                flags f,
                properties<C> const& p)
        {
          _parse_ (
            std::basic_string<C> (uri), root_namespace, root_name, sax, f, p);
          return post ();
        }


        // parse (istream)
        //

        template <typename X, typename C>
        X traverser<X, C>::
        _parse (std::istream& is,
                std::basic_string<C> const& root_namespace,
                std::basic_string<C> const& root_name,
                flags f,
                properties<C> const& p)
        {
          _parse_ (is, root_namespace, root_name, f, p);
          return post ();
        }

        template <typename X, typename C>
        X traverser<X, C>::
        _parse (std::istream& is,
                std::basic_string<C> const& root_namespace,
                std::basic_string<C> const& root_name,
                xml::error_handler<C>& eh,
                flags f,
                properties<C> const& p)
        {
          _parse_ (is, root_namespace, root_name, eh, f, p);
          return post ();
        }

        template <typename X, typename C>
        X traverser<X, C>::
        _parse (std::istream& is,
                std::basic_string<C> const& root_namespace,
                std::basic_string<C> const& root_name,
                xercesc::ErrorHandler& eh,
                flags f,
                properties<C> const& p)
        {
          _parse_ (is, root_namespace, root_name, eh, f, p);
          return post ();
        }

        template <typename X, typename C>
        X traverser<X, C>::
        _parse (std::istream& is,
                std::basic_string<C> const& root_namespace,
                std::basic_string<C> const& root_name,
                xercesc::SAX2XMLReader& sax,
                flags f,
                properties<C> const& p)
        {
          _parse_ (is, root_namespace, root_name, sax, f, p);
          return post ();
        }


        // parse (istream, system_id)
        //


        template <typename X, typename C>
        X traverser<X, C>::
        _parse (std::istream& is,
                std::basic_string<C> const& sys_id,
                std::basic_string<C> const& root_namespace,
                std::basic_string<C> const& root_name,
                flags f,
                properties<C> const& p)
        {
          _parse_ (is, sys_id, root_namespace, root_name, f, p);
          return post ();
        }

        template <typename X, typename C>
        X traverser<X, C>::
        _parse (std::istream& is,
                std::basic_string<C> const& sys_id,
                std::basic_string<C> const& root_namespace,
                std::basic_string<C> const& root_name,
                xml::error_handler<C>& eh,
                flags f,
                properties<C> const& p)
        {
          _parse_ (is, sys_id, root_namespace, root_name, eh, f, p);
          return post ();
        }

        template <typename X, typename C>
        X traverser<X, C>::
        _parse (std::istream& is,
                std::basic_string<C> const& sys_id,
                std::basic_string<C> const& root_namespace,
                std::basic_string<C> const& root_name,
                xercesc::ErrorHandler& eh,
                flags f,
                properties<C> const& p)
        {
          _parse_ (is, sys_id, root_namespace, root_name, eh, f, p);
          return post ();
        }

        template <typename X, typename C>
        X traverser<X, C>::
        _parse (std::istream& is,
                std::basic_string<C> const& sys_id,
                std::basic_string<C> const& root_namespace,
                std::basic_string<C> const& root_name,
                xercesc::SAX2XMLReader& sax,
                flags f,
                properties<C> const& p)
        {
          _parse_ (is, sys_id, root_namespace, root_name, sax, f, p);
          return post ();
        }


        // parse (istream, system_id, public_id)
        //

        template <typename X, typename C>
        X traverser<X, C>::
        _parse (std::istream& is,
                std::basic_string<C> const& sys_id,
                std::basic_string<C> const& pub_id,
                std::basic_string<C> const& root_namespace,
                std::basic_string<C> const& root_name,
                flags f,
                properties<C> const& p)
        {
          _parse_ (is, sys_id, pub_id, root_namespace, root_name, f, p);
          return post ();
        }

        template <typename X, typename C>
        X traverser<X, C>::
        _parse (std::istream& is,
                std::basic_string<C> const& sys_id,
                std::basic_string<C> const& pub_id,
                std::basic_string<C> const& root_namespace,
                std::basic_string<C> const& root_name,
                xml::error_handler<C>& eh,
                flags f,
                properties<C> const& p)
        {
          _parse_ (is, sys_id, pub_id, root_namespace, root_name, eh, f, p);
          return post ();
        }

        template <typename X, typename C>
        X traverser<X, C>::
        _parse (std::istream& is,
                std::basic_string<C> const& sys_id,
                std::basic_string<C> const& pub_id,
                std::basic_string<C> const& root_namespace,
                std::basic_string<C> const& root_name,
                xercesc::ErrorHandler& eh,
                flags f,
                properties<C> const& p)
        {
          _parse_ (is, sys_id, pub_id, root_namespace, root_name, eh, f, p);
          return post ();
        }

        template <typename X, typename C>
        X traverser<X, C>::
        _parse (std::istream& is,
                std::basic_string<C> const& sys_id,
                std::basic_string<C> const& pub_id,
                std::basic_string<C> const& root_namespace,
                std::basic_string<C> const& root_name,
                xercesc::SAX2XMLReader& sax,
                flags f,
                properties<C> const& p)
        {
          _parse_ (is, sys_id, pub_id, root_namespace, root_name, sax, f, p);
          return post ();
        }


        // parse (InputSource)
        //


        template <typename X, typename C>
        X traverser<X, C>::
        _parse (xercesc::InputSource const& is,
                std::basic_string<C> const& root_namespace,
                std::basic_string<C> const& root_name,
                flags f,
                properties<C> const& p)
        {
          _parse_ (is, root_namespace, root_name, f, p);
          return post ();
        }

        template <typename X, typename C>
        X traverser<X, C>::
        _parse (xercesc::InputSource const& is,
                std::basic_string<C> const& root_namespace,
                std::basic_string<C> const& root_name,
                xml::error_handler<C>& eh,
                flags f,
                properties<C> const& p)
        {
          _parse_ (is, root_namespace, root_name, eh, f, p);
          return post ();
        }

        template <typename X, typename C>
        X traverser<X, C>::
        _parse (xercesc::InputSource const& is,
                std::basic_string<C> const& root_namespace,
                std::basic_string<C> const& root_name,
                xercesc::ErrorHandler& eh,
                flags f,
                properties<C> const& p)
        {
          _parse_ (is, root_namespace, root_name, eh, f, p);
          return post ();
        }


        template <typename X, typename C>
        X traverser<X, C>::
        _parse (xercesc::InputSource const& is,
                std::basic_string<C> const& root_namespace,
                std::basic_string<C> const& root_name,
                xercesc::SAX2XMLReader& sax,
                flags f,
                properties<C> const& p)
        {
          _parse_ (is, root_namespace, root_name, sax, f, p);
          return post ();
        }


        // traverser<void>
        //

        // parse (uri)
        //
        template <typename C>
        void traverser<void, C>::
        _parse (std::basic_string<C> const& uri,
                std::basic_string<C> const& root_namespace,
                std::basic_string<C> const& root_name,
                flags f,
                properties<C> const& p)
        {
          _parse_ (uri, root_namespace, root_name, f, p);
          post ();
        }


        template <typename C>
        void traverser<void, C>::
        _parse (C const* uri,
                std::basic_string<C> const& root_namespace,
                std::basic_string<C> const& root_name,
                flags f,
                properties<C> const& p)
        {
          _parse_ (
            std::basic_string<C> (uri), root_namespace, root_name, f, p);
          post ();
        }

        // error_handler
        //

        template <typename C>
        void traverser<void, C>::
        _parse (std::basic_string<C> const& uri,
                std::basic_string<C> const& root_namespace,
                std::basic_string<C> const& root_name,
                xml::error_handler<C>& eh,
                flags f,
                properties<C> const& p)
        {
          _parse_ (uri, root_namespace, root_name, eh, f, p);
          post ();
        }


        template <typename C>
        void traverser<void, C>::
        _parse (C const* uri,
                std::basic_string<C> const& root_namespace,
                std::basic_string<C> const& root_name,
                xml::error_handler<C>& eh,
                flags f,
                properties<C> const& p)
        {
          _parse_ (
            std::basic_string<C> (uri), root_namespace, root_name, eh, f, p);

          post ();
        }

        // ErrorHandler
        //

        template <typename C>
        void traverser<void, C>::
        _parse (std::basic_string<C> const& uri,
                std::basic_string<C> const& root_namespace,
                std::basic_string<C> const& root_name,
                xercesc::ErrorHandler& eh,
                flags f,
                properties<C> const& p)
        {
          _parse_ (uri, root_namespace, root_name, eh, f, p);
          post ();
        }


        template <typename C>
        void traverser<void, C>::
        _parse (C const* uri,
                std::basic_string<C> const& root_namespace,
                std::basic_string<C> const& root_name,
                xercesc::ErrorHandler& eh,
                flags f,
                properties<C> const& p)
        {
          _parse_ (
            std::basic_string<C> (uri), root_namespace, root_name, eh, f, p);
          post ();
        }

        // SAX2XMLReader
        //

        template <typename C>
        void traverser<void, C>::
        _parse (std::basic_string<C> const& uri,
                std::basic_string<C> const& root_namespace,
                std::basic_string<C> const& root_name,
                xercesc::SAX2XMLReader& sax,
                flags f,
                properties<C> const& p)
        {
          _parse_ (uri, root_namespace, root_name, sax, f, p);
          post ();
        }

        template <typename C>
        void traverser<void, C>::
        _parse (C const* uri,
                std::basic_string<C> const& root_namespace,
                std::basic_string<C> const& root_name,
                xercesc::SAX2XMLReader& sax,
                flags f,
                properties<C> const& p)
        {
          _parse_ (
            std::basic_string<C> (uri), root_namespace, root_name, sax, f, p);
          post ();
        }


        // parse (istream)
        //

        template <typename C>
        void traverser<void, C>::
        _parse (std::istream& is,
                std::basic_string<C> const& root_namespace,
                std::basic_string<C> const& root_name,
                flags f,
                properties<C> const& p)
        {
          _parse_ (is, root_namespace, root_name, f, p);
          post ();
        }

        template <typename C>
        void traverser<void, C>::
        _parse (std::istream& is,
                std::basic_string<C> const& root_namespace,
                std::basic_string<C> const& root_name,
                xml::error_handler<C>& eh,
                flags f,
                properties<C> const& p)
        {
          _parse_ (is, root_namespace, root_name, eh, f, p);
          post ();
        }

        template <typename C>
        void traverser<void, C>::
        _parse (std::istream& is,
                std::basic_string<C> const& root_namespace,
                std::basic_string<C> const& root_name,
                xercesc::ErrorHandler& eh,
                flags f,
                properties<C> const& p)
        {
          _parse_ (is, root_namespace, root_name, eh, f, p);
          post ();
        }

        template <typename C>
        void traverser<void, C>::
        _parse (std::istream& is,
                std::basic_string<C> const& root_namespace,
                std::basic_string<C> const& root_name,
                xercesc::SAX2XMLReader& sax,
                flags f,
                properties<C> const& p)
        {
          _parse_ (is, root_namespace, root_name, sax, f, p);
          post ();
        }


        // parse (istream, system_id)
        //


        template <typename C>
        void traverser<void, C>::
        _parse (std::istream& is,
                std::basic_string<C> const& sys_id,
                std::basic_string<C> const& root_namespace,
                std::basic_string<C> const& root_name,
                flags f,
                properties<C> const& p)
        {
          _parse_ (is, sys_id, root_namespace, root_name, f, p);
          post ();
        }

        template <typename C>
        void traverser<void, C>::
        _parse (std::istream& is,
                std::basic_string<C> const& sys_id,
                std::basic_string<C> const& root_namespace,
                std::basic_string<C> const& root_name,
                xml::error_handler<C>& eh,
                flags f,
                properties<C> const& p)
        {
          _parse_ (is, sys_id, root_namespace, root_name, eh, f, p);
          post ();
        }

        template <typename C>
        void traverser<void, C>::
        _parse (std::istream& is,
                std::basic_string<C> const& sys_id,
                std::basic_string<C> const& root_namespace,
                std::basic_string<C> const& root_name,
                xercesc::ErrorHandler& eh,
                flags f,
                properties<C> const& p)
        {
          _parse_ (is, sys_id, root_namespace, root_name, eh, f, p);
          post ();
        }

        template <typename C>
        void traverser<void, C>::
        _parse (std::istream& is,
                std::basic_string<C> const& sys_id,
                std::basic_string<C> const& root_namespace,
                std::basic_string<C> const& root_name,
                xercesc::SAX2XMLReader& sax,
                flags f,
                properties<C> const& p)
        {
          _parse_ (is, sys_id, root_namespace, root_name, sax, f, p);
          post ();
        }


        // parse (istream, system_id, public_id)
        //

        template <typename C>
        void traverser<void, C>::
        _parse (std::istream& is,
                std::basic_string<C> const& sys_id,
                std::basic_string<C> const& pub_id,
                std::basic_string<C> const& root_namespace,
                std::basic_string<C> const& root_name,
                flags f,
                properties<C> const& p)
        {
          _parse_ (is, sys_id, pub_id, root_namespace, root_name, f, p);
          post ();
        }

        template <typename C>
        void traverser<void, C>::
        _parse (std::istream& is,
                std::basic_string<C> const& sys_id,
                std::basic_string<C> const& pub_id,
                std::basic_string<C> const& root_namespace,
                std::basic_string<C> const& root_name,
                xml::error_handler<C>& eh,
                flags f,
                properties<C> const& p)
        {
          _parse_ (is, sys_id, pub_id, root_namespace, root_name, eh, f, p);
          post ();
        }

        template <typename C>
        void traverser<void, C>::
        _parse (std::istream& is,
                std::basic_string<C> const& sys_id,
                std::basic_string<C> const& pub_id,
                std::basic_string<C> const& root_namespace,
                std::basic_string<C> const& root_name,
                xercesc::ErrorHandler& eh,
                flags f,
                properties<C> const& p)
        {
          _parse_ (is, sys_id, pub_id, root_namespace, root_name, eh, f, p);
          post ();
        }

        template <typename C>
        void traverser<void, C>::
        _parse (std::istream& is,
                std::basic_string<C> const& sys_id,
                std::basic_string<C> const& pub_id,
                std::basic_string<C> const& root_namespace,
                std::basic_string<C> const& root_name,
                xercesc::SAX2XMLReader& sax,
                flags f,
                properties<C> const& p)
        {
          _parse_ (is, sys_id, pub_id, root_namespace, root_name, sax, f, p);
          post ();
        }


        // parse (InputSource)
        //


        template <typename C>
        void traverser<void, C>::
        _parse (xercesc::InputSource const& is,
                std::basic_string<C> const& root_namespace,
                std::basic_string<C> const& root_name,
                flags f,
                properties<C> const& p)
        {
          _parse_ (is, root_namespace, root_name, f, p);
          post ();
        }

        template <typename C>
        void traverser<void, C>::
        _parse (xercesc::InputSource const& is,
                std::basic_string<C> const& root_namespace,
                std::basic_string<C> const& root_name,
                xml::error_handler<C>& eh,
                flags f,
                properties<C> const& p)
        {
          _parse_ (is, root_namespace, root_name, eh, f, p);
          post ();
        }

        template <typename C>
        void traverser<void, C>::
        _parse (xercesc::InputSource const& is,
                std::basic_string<C> const& root_namespace,
                std::basic_string<C> const& root_name,
                xercesc::ErrorHandler& eh,
                flags f,
                properties<C> const& p)
        {
          _parse_ (is, root_namespace, root_name, eh, f, p);
          post ();
        }


        template <typename C>
        void traverser<void, C>::
        _parse (xercesc::InputSource const& is,
                std::basic_string<C> const& root_namespace,
                std::basic_string<C> const& root_name,
                xercesc::SAX2XMLReader& sax,
                flags f,
                properties<C> const& p)
        {
          _parse_ (is, root_namespace, root_name, sax, f, p);
          post ();
        }
      }
    }
  }
}
