// file      : xsd/cxx/parser/xerces/elements.hxx
// author    : Boris Kolpackov <boris@codesynthesis.com>
// copyright : Copyright (c) 2005-2006 Code Synthesis Tools CC
// license   : GNU GPL v2 + exceptions; see accompanying LICENSE file

#ifndef XSD_CXX_PARSER_XERCES_ELEMENTS_HXX
#define XSD_CXX_PARSER_XERCES_ELEMENTS_HXX

#include <memory>  // std::auto_ptr
#include <string>
#include <iosfwd>

#include <xercesc/sax/InputSource.hpp>
#include <xercesc/sax2/SAX2XMLReader.hpp>
#include <xercesc/sax2/DefaultHandler.hpp>

#include <xsd/cxx/xml/elements.hxx>
#include <xsd/cxx/xml/error-handler.hxx>

#include <xsd/cxx/parser/exceptions.hxx>
#include <xsd/cxx/parser/elements.hxx>

namespace xsd
{
  namespace cxx
  {
    namespace parser
    {
      namespace xerces
      {
        //
        //
        struct flags
        {
          // Use the following flags to modify the default behavior
          // of the parsing functions.
          //

          // Do not try to validate instance documents. Note that
          // the xsd runtime assumes instance documents are valid so
          // you had better make sure they are if you specify this flag.
          //
          //
          static unsigned long const dont_validate = 0x00000001;

          // Do not initialize the Xerces-C++ runtime.
          //
          static unsigned long const dont_initialize = 0x00000002;

        public:
          flags (unsigned long x = 0)
              : x_ (x)
          {
          }

          operator unsigned long () const
          {
            return x_;
          }

        private:
          unsigned long x_;
        };


        // Parsing properties. Refer to xsd/cxx/xml/elements.hxx for
        // XML-related properties.
        //
        template <typename C>
        class properties: public xml::properties<C>
        {
        };


        // Common code before generic and void-specialization split.
        //
        template <typename C>
        struct traverser_common: virtual traverser_impl<C>
        {
          // Parse URI or a local file.
          //
        protected:
          void
          _parse_ (std::basic_string<C> const& uri,
                   std::basic_string<C> const& root_element_namespace,
                   std::basic_string<C> const& root_element_name,
                   flags,
                   properties<C> const&);

          // With error_handler.
          //

          void
          _parse_ (std::basic_string<C> const& uri,
                   std::basic_string<C> const& root_element_namespace,
                   std::basic_string<C> const& root_element_name,
                   xml::error_handler<C>&,
                   flags,
                   properties<C> const&);

          // With ErrorHandler.
          //

          void
          _parse_ (std::basic_string<C> const& uri,
                   std::basic_string<C> const& root_element_namespace,
                   std::basic_string<C> const& root_element_name,
                   xercesc::ErrorHandler&,
                   flags,
                   properties<C> const&);

          // Using SAX2XMLReader.
          //

          void
          _parse_ (std::basic_string<C> const& uri,
                   std::basic_string<C> const& root_element_namespace,
                   std::basic_string<C> const& root_element_name,
                   xercesc::SAX2XMLReader&,
                   flags,
                   properties<C> const&);


          // Parse std::istream.
          //
        protected:
          void
          _parse_ (std::istream&,
                   std::basic_string<C> const& root_element_namespace,
                   std::basic_string<C> const& root_element_name,
                   flags,
                   properties<C> const&);

          // With error_handler.
          //

          void
          _parse_ (std::istream&,
                   std::basic_string<C> const& root_element_namespace,
                   std::basic_string<C> const& root_element_name,
                   xml::error_handler<C>&,
                   flags,
                   properties<C> const&);

          // With ErrorHandler.
          //

          void
          _parse_ (std::istream&,
                   std::basic_string<C> const& root_element_namespace,
                   std::basic_string<C> const& root_element_name,
                   xercesc::ErrorHandler&,
                   flags,
                   properties<C> const&);

          // Using SAX2XMLReader.
          //

          void
          _parse_ (std::istream&,
                   std::basic_string<C> const& root_element_namespace,
                   std::basic_string<C> const& root_element_name,
                   xercesc::SAX2XMLReader&,
                   flags,
                   properties<C> const&);


          // Parse std::istream with a system id.
          //
        protected:
          void
          _parse_ (std::istream&,
                   std::basic_string<C> const& system_id,
                   std::basic_string<C> const& root_element_namespace,
                   std::basic_string<C> const& root_element_name,
                   flags,
                   properties<C> const&);

          // With error_handler.
          //

          void
          _parse_ (std::istream&,
                   std::basic_string<C> const& system_id,
                   std::basic_string<C> const& root_element_namespace,
                   std::basic_string<C> const& root_element_name,
                   xml::error_handler<C>&,
                   flags,
                   properties<C> const&);

          // With ErrorHandler.
          //

          void
          _parse_ (std::istream&,
                   std::basic_string<C> const& system_id,
                   std::basic_string<C> const& root_element_namespace,
                   std::basic_string<C> const& root_element_name,
                   xercesc::ErrorHandler&,
                   flags,
                   properties<C> const&);

          // Using SAX2XMLReader.
          //

          void
          _parse_ (std::istream&,
                   std::basic_string<C> const& system_id,
                   std::basic_string<C> const& root_element_namespace,
                   std::basic_string<C> const& root_element_name,
                   xercesc::SAX2XMLReader&,
                   flags,
                   properties<C> const&);



          // Parse std::istream with system and public ids.
          //
        protected:
          void
          _parse_ (std::istream&,
                   std::basic_string<C> const& system_id,
                   std::basic_string<C> const& public_id,
                   std::basic_string<C> const& root_element_namespace,
                   std::basic_string<C> const& root_element_name,
                   flags,
                   properties<C> const&);

          // With error_handler.
          //

          void
          _parse_ (std::istream&,
                   std::basic_string<C> const& system_id,
                   std::basic_string<C> const& public_id,
                   std::basic_string<C> const& root_element_namespace,
                   std::basic_string<C> const& root_element_name,
                   xml::error_handler<C>&,
                   flags,
                   properties<C> const&);

          // With ErrorHandler.
          //

          void
          _parse_ (std::istream&,
                   std::basic_string<C> const& system_id,
                   std::basic_string<C> const& public_id,
                   std::basic_string<C> const& root_element_namespace,
                   std::basic_string<C> const& root_element_name,
                   xercesc::ErrorHandler&,
                   flags,
                   properties<C> const&);

          // Using SAX2XMLReader.
          //

          void
          _parse_ (std::istream&,
                   std::basic_string<C> const& system_id,
                   std::basic_string<C> const& public_id,
                   std::basic_string<C> const& root_element_namespace,
                   std::basic_string<C> const& root_element_name,
                   xercesc::SAX2XMLReader&,
                   flags,
                   properties<C> const&);


          // Parse InputSource.
          //
        protected:
          void
          _parse_ (xercesc::InputSource const&,
                   std::basic_string<C> const& root_element_namespace,
                   std::basic_string<C> const& root_element_name,
                   flags,
                   properties<C> const&);

          // With error_handler.
          //

          void
          _parse_ (xercesc::InputSource const&,
                   std::basic_string<C> const& root_element_namespace,
                   std::basic_string<C> const& root_element_name,
                   xml::error_handler<C>&,
                   flags,
                   properties<C> const&);

          // With ErrorHandler.
          //

          void
          _parse_ (xercesc::InputSource const&,
                   std::basic_string<C> const& root_element_namespace,
                   std::basic_string<C> const& root_element_name,
                   xercesc::ErrorHandler&,
                   flags,
                   properties<C> const&);

          // Using SAX2XMLReader.
          //

          void
          _parse_ (xercesc::InputSource const&,
                   std::basic_string<C> const& root_element_namespace,
                   std::basic_string<C> const& root_element_name,
                   xercesc::SAX2XMLReader&,
                   flags,
                   properties<C> const&);


        private:
          void
          _parse_ (std::basic_string<C> const& uri,
                   std::basic_string<C> const& root_element_namespace,
                   std::basic_string<C> const& root_element_name,
                   xercesc::ErrorHandler&,
                   xercesc::SAX2XMLReader&,
                   flags,
                   properties<C> const&);

          void
          _parse_ (xercesc::InputSource const&,
                   std::basic_string<C> const& root_element_namespace,
                   std::basic_string<C> const& root_element_name,
                   xercesc::ErrorHandler&,
                   xercesc::SAX2XMLReader&,
                   flags,
                   properties<C> const&);

        private:
          std::auto_ptr<xercesc::SAX2XMLReader>
          _create_sax (flags, properties<C> const&);
        };


        //
        //
        template <typename X, typename C>
        struct traverser: virtual traverser_common<C>
        {
          virtual X
          post () = 0;


        public:
          // Parse URI or a local file. We have to overload it for C const*
          // bacause xercesc::InputSource has an implicit constructor that
          // takes char const*.
          //
          X
          _parse (std::basic_string<C> const& uri,
                  std::basic_string<C> const& root_element_namespace,
                  std::basic_string<C> const& root_element_name,
                  flags = 0,
                  properties<C> const& = properties<C> ());

          X
          _parse (C const* uri,
                  std::basic_string<C> const& root_element_namespace,
                  std::basic_string<C> const& root_element_name,
                  flags = 0,
                  properties<C> const& = properties<C> ());


          // Parse URI or a local file with a user-provided error_handler
          // object.
          //
          X
          _parse (std::basic_string<C> const& uri,
                  std::basic_string<C> const& root_element_namespace,
                  std::basic_string<C> const& root_element_name,
                  xml::error_handler<C>&,
                  flags = 0,
                  properties<C> const& = properties<C> ());

          X
          _parse (C const* uri,
                  std::basic_string<C> const& root_element_namespace,
                  std::basic_string<C> const& root_element_name,
                  xml::error_handler<C>&,
                  flags = 0,
                  properties<C> const& = properties<C> ());


          // Parse URI or a local file with a user-provided ErrorHandler
          // object. Note that you must initialize the Xerces-C++ runtime
          // before calling these functions.
          //
          X
          _parse (std::basic_string<C> const& uri,
                  std::basic_string<C> const& root_element_namespace,
                  std::basic_string<C> const& root_element_name,
                  xercesc::ErrorHandler&,
                  flags = 0,
                  properties<C> const& = properties<C> ());

          X
          _parse (C const* uri,
                  std::basic_string<C> const& root_element_namespace,
                  std::basic_string<C> const& root_element_name,
                  xercesc::ErrorHandler&,
                  flags = 0,
                  properties<C> const& = properties<C> ());


          // Parse URI or a local file using a user-provided SAX2XMLReader
          // object. Note that you must initialize the Xerces-C++ runtime
          // before calling these functions.
          //
          X
          _parse (std::basic_string<C> const& uri,
                  std::basic_string<C> const& root_element_namespace,
                  std::basic_string<C> const& root_element_name,
                  xercesc::SAX2XMLReader&,
                  flags = 0,
                  properties<C> const& = properties<C> ());

          X
          _parse (C const* uri,
                  std::basic_string<C> const& root_element_namespace,
                  std::basic_string<C> const& root_element_name,
                  xercesc::SAX2XMLReader&,
                  flags = 0,
                  properties<C> const& = properties<C> ());


        public:
          // System id is a "system" identifier of the resources (e.g.,
          // URI or a full file path). Public id is a "public" identifier
          // of the resource (e.g., application-specific name or relative
          // file path). System id is used to resolve relative paths.
          // In diagnostics messages system id is used if public id is
          // not available. Otherwise public id is used.
          //
          // Note that these sematics are open for interpretation. Both
          // system and public ids are optional and you can view them as
          // a way to associate two strings with the stream being parsed
          // which are made available to you in error handlers.
          //


          // Parse std::istream.
          //
          X
          _parse (std::istream&,
                  std::basic_string<C> const& root_element_namespace,
                  std::basic_string<C> const& root_element_name,
                  flags = 0,
                  properties<C> const& = properties<C> ());


          // Parse std::istream with user-provided error_handler object.
          //
          X
          _parse (std::istream&,
                  std::basic_string<C> const& root_element_namespace,
                  std::basic_string<C> const& root_element_name,
                  xml::error_handler<C>&,
                  flags = 0,
                  properties<C> const& = properties<C> ());


          // Parse std::istream with user-provided ErrorHandler object.
          // Note that you must initialize the Xerces-C++ runtime before
          // calling this function.
          //
          X
          _parse (std::istream&,
                  std::basic_string<C> const& root_element_namespace,
                  std::basic_string<C> const& root_element_name,
                  xercesc::ErrorHandler&,
                  flags = 0,
                  properties<C> const& = properties<C> ());


          // Parse std::istream using user-provided SAX2XMLReader object.
          // Note that you must initialize the Xerces-C++ runtime before
          // calling this function.
          //
          X
          _parse (std::istream&,
                  std::basic_string<C> const& root_element_namespace,
                  std::basic_string<C> const& root_element_name,
                  xercesc::SAX2XMLReader&,
                  flags = 0,
                  properties<C> const& = properties<C> ());


        public:
          // Parse std::istream with a system id.
          //
          X
          _parse (std::istream&,
                  std::basic_string<C> const& system_id,
                  std::basic_string<C> const& root_element_namespace,
                  std::basic_string<C> const& root_element_name,
                  flags = 0,
                  properties<C> const& = properties<C> ());


          // Parse std::istream with a system id and a user-provided
          // error_handler object.
          //
          X
          _parse (std::istream&,
                  std::basic_string<C> const& system_id,
                  std::basic_string<C> const& root_element_namespace,
                  std::basic_string<C> const& root_element_name,
                  xml::error_handler<C>&,
                  flags = 0,
                  properties<C> const& = properties<C> ());


          // Parse std::istream with a system id and a user-provided
          // ErrorHandler object. Note that you must initialize the
          // Xerces-C++ runtime before calling this function.
          //
          X
          _parse (std::istream&,
                  std::basic_string<C> const& system_id,
                  std::basic_string<C> const& root_element_namespace,
                  std::basic_string<C> const& root_element_name,
                  xercesc::ErrorHandler&,
                  flags = 0,
                  properties<C> const& = properties<C> ());


          // Parse std::istream with a system id using a user-provided
          // SAX2XMLReader object. Note that you must initialize the
          // Xerces-C++ runtime before calling this function.
          //
          X
          _parse (std::istream&,
                  std::basic_string<C> const& system_id,
                  std::basic_string<C> const& root_element_namespace,
                  std::basic_string<C> const& root_element_name,
                  xercesc::SAX2XMLReader&,
                  flags = 0,
                  properties<C> const& = properties<C> ());



        public:
          // Parse std::istream with system and public ids.
          //
          X
          _parse (std::istream&,
                  std::basic_string<C> const& system_id,
                  std::basic_string<C> const& public_id,
                  std::basic_string<C> const& root_element_namespace,
                  std::basic_string<C> const& root_element_name,
                  flags = 0,
                  properties<C> const& = properties<C> ());


          // Parse std::istream with system and public ids and a user-provided
          // error_handler object.
          //
          X
          _parse (std::istream&,
                  std::basic_string<C> const& system_id,
                  std::basic_string<C> const& public_id,
                  std::basic_string<C> const& root_element_namespace,
                  std::basic_string<C> const& root_element_name,
                  xml::error_handler<C>&,
                  flags = 0,
                  properties<C> const& = properties<C> ());


          // Parse std::istream with system and public ids and a user-provided
          // ErrorHandler object. Note that you must initialize the Xerces-C++
          // runtime before calling this function.
          //
          X
          _parse (std::istream&,
                  std::basic_string<C> const& system_id,
                  std::basic_string<C> const& public_id,
                  std::basic_string<C> const& root_element_namespace,
                  std::basic_string<C> const& root_element_name,
                  xercesc::ErrorHandler&,
                  flags = 0,
                  properties<C> const& = properties<C> ());


          // Parse std::istream with system and public ids using a user-
          // provided SAX2XMLReader object. Note that you must initialize
          // the Xerces-C++ runtime before calling this function.
          //
          X
          _parse (std::istream&,
                  std::basic_string<C> const& system_id,
                  std::basic_string<C> const& public_id,
                  std::basic_string<C> const& root_element_namespace,
                  std::basic_string<C> const& root_element_name,
                  xercesc::SAX2XMLReader&,
                  flags = 0,
                  properties<C> const& = properties<C> ());


        public:
          // Parse InputSource. Note that you must initialize the Xerces-C++
          // runtime before calling this function.
          //
          X
          _parse (xercesc::InputSource const&,
                  std::basic_string<C> const& root_element_namespace,
                  std::basic_string<C> const& root_element_name,
                  flags = 0,
                  properties<C> const& = properties<C> ());


          // Parse InputSource with a user-provided error_handler object.
          // Note that you must initialize the Xerces-C++ runtime before
          // calling this function.
          //
          X
          _parse (xercesc::InputSource const&,
                  std::basic_string<C> const& root_element_namespace,
                  std::basic_string<C> const& root_element_name,
                  xml::error_handler<C>&,
                  flags = 0,
                  properties<C> const& = properties<C> ());


          // Parse InputSource with a user-provided ErrorHandler object.
          // Note that you must initialize the Xerces-C++ runtime before
          // calling this function.
          //
          X
          _parse (xercesc::InputSource const&,
                  std::basic_string<C> const& root_element_namespace,
                  std::basic_string<C> const& root_element_name,
                  xercesc::ErrorHandler&,
                  flags = 0,
                  properties<C> const& = properties<C> ());


          // Parse InputSource using a user-provided SAX2XMLReader object.
          // Note that you must initialize the Xerces-C++ runtime before
          // calling this function.
          //
          X
          _parse (xercesc::InputSource const&,
                  std::basic_string<C> const& root_element_namespace,
                  std::basic_string<C> const& root_element_name,
                  xercesc::SAX2XMLReader&,
                  flags = 0,
                  properties<C> const& = properties<C> ());
        };


        // Specialization for void.
        //
        template <typename C>
        struct traverser<void, C>: virtual traverser_common<C>
        {
          virtual void
          post ()
          {
          }

        public:
          // Parse URI or a local file. We have to overload it for C const*
          // bacause xercesc::InputSource has an implicit constructor that
          // takes char const*.
          //
          void
          _parse (std::basic_string<C> const& uri,
                  std::basic_string<C> const& root_element_namespace,
                  std::basic_string<C> const& root_element_name,
                  flags = 0,
                  properties<C> const& = properties<C> ());

          void
          _parse (C const* uri,
                  std::basic_string<C> const& root_element_namespace,
                  std::basic_string<C> const& root_element_name,
                  flags = 0,
                  properties<C> const& = properties<C> ());

          void
          _parse (std::basic_string<C> const& uri,
                  std::basic_string<C> const& root_element_namespace,
                  std::basic_string<C> const& root_element_name,
                  xml::error_handler<C>&,
                  flags = 0,
                  properties<C> const& = properties<C> ());

          void
          _parse (C const* uri,
                  std::basic_string<C> const& root_element_namespace,
                  std::basic_string<C> const& root_element_name,
                  xml::error_handler<C>&,
                  flags = 0,
                  properties<C> const& = properties<C> ());

          void
          _parse (std::basic_string<C> const& uri,
                  std::basic_string<C> const& root_element_namespace,
                  std::basic_string<C> const& root_element_name,
                  xercesc::ErrorHandler&,
                  flags = 0,
                  properties<C> const& = properties<C> ());

          void
          _parse (C const* uri,
                  std::basic_string<C> const& root_element_namespace,
                  std::basic_string<C> const& root_element_name,
                  xercesc::ErrorHandler&,
                  flags = 0,
                  properties<C> const& = properties<C> ());

          void
          _parse (std::basic_string<C> const& uri,
                  std::basic_string<C> const& root_element_namespace,
                  std::basic_string<C> const& root_element_name,
                  xercesc::SAX2XMLReader&,
                  flags = 0,
                  properties<C> const& = properties<C> ());

          void
          _parse (C const* uri,
                  std::basic_string<C> const& root_element_namespace,
                  std::basic_string<C> const& root_element_name,
                  xercesc::SAX2XMLReader&,
                  flags = 0,
                  properties<C> const& = properties<C> ());


        public:
          // Parse std::istream.
          //
          void
          _parse (std::istream&,
                  std::basic_string<C> const& root_element_namespace,
                  std::basic_string<C> const& root_element_name,
                  flags = 0,
                  properties<C> const& = properties<C> ());

          void
          _parse (std::istream&,
                  std::basic_string<C> const& root_element_namespace,
                  std::basic_string<C> const& root_element_name,
                  xml::error_handler<C>&,
                  flags = 0,
                  properties<C> const& = properties<C> ());

          void
          _parse (std::istream&,
                  std::basic_string<C> const& root_element_namespace,
                  std::basic_string<C> const& root_element_name,
                  xercesc::ErrorHandler&,
                  flags = 0,
                  properties<C> const& = properties<C> ());

          void
          _parse (std::istream&,
                  std::basic_string<C> const& root_element_namespace,
                  std::basic_string<C> const& root_element_name,
                  xercesc::SAX2XMLReader&,
                  flags = 0,
                  properties<C> const& = properties<C> ());


        public:
          // Parse std::istream with a system id.
          //
          void
          _parse (std::istream&,
                  std::basic_string<C> const& system_id,
                  std::basic_string<C> const& root_element_namespace,
                  std::basic_string<C> const& root_element_name,
                  flags = 0,
                  properties<C> const& = properties<C> ());

          void
          _parse (std::istream&,
                  std::basic_string<C> const& system_id,
                  std::basic_string<C> const& root_element_namespace,
                  std::basic_string<C> const& root_element_name,
                  xml::error_handler<C>&,
                  flags = 0,
                  properties<C> const& = properties<C> ());

          void
          _parse (std::istream&,
                  std::basic_string<C> const& system_id,
                  std::basic_string<C> const& root_element_namespace,
                  std::basic_string<C> const& root_element_name,
                  xercesc::ErrorHandler&,
                  flags = 0,
                  properties<C> const& = properties<C> ());

          void
          _parse (std::istream&,
                  std::basic_string<C> const& system_id,
                  std::basic_string<C> const& root_element_namespace,
                  std::basic_string<C> const& root_element_name,
                  xercesc::SAX2XMLReader&,
                  flags = 0,
                  properties<C> const& = properties<C> ());

        public:
          // Parse std::istream with system and public ids.
          //
          void
          _parse (std::istream&,
                  std::basic_string<C> const& system_id,
                  std::basic_string<C> const& public_id,
                  std::basic_string<C> const& root_element_namespace,
                  std::basic_string<C> const& root_element_name,
                  flags = 0,
                  properties<C> const& = properties<C> ());

          void
          _parse (std::istream&,
                  std::basic_string<C> const& system_id,
                  std::basic_string<C> const& public_id,
                  std::basic_string<C> const& root_element_namespace,
                  std::basic_string<C> const& root_element_name,
                  xml::error_handler<C>&,
                  flags = 0,
                  properties<C> const& = properties<C> ());

          void
          _parse (std::istream&,
                  std::basic_string<C> const& system_id,
                  std::basic_string<C> const& public_id,
                  std::basic_string<C> const& root_element_namespace,
                  std::basic_string<C> const& root_element_name,
                  xercesc::ErrorHandler&,
                  flags = 0,
                  properties<C> const& = properties<C> ());

          void
          _parse (std::istream&,
                  std::basic_string<C> const& system_id,
                  std::basic_string<C> const& public_id,
                  std::basic_string<C> const& root_element_namespace,
                  std::basic_string<C> const& root_element_name,
                  xercesc::SAX2XMLReader&,
                  flags = 0,
                  properties<C> const& = properties<C> ());


        public:
          // Parse InputSource.
          //
          void
          _parse (xercesc::InputSource const&,
                  std::basic_string<C> const& root_element_namespace,
                  std::basic_string<C> const& root_element_name,
                  flags = 0,
                  properties<C> const& = properties<C> ());

          void
          _parse (xercesc::InputSource const&,
                  std::basic_string<C> const& root_element_namespace,
                  std::basic_string<C> const& root_element_name,
                  xml::error_handler<C>&,
                  flags = 0,
                  properties<C> const& = properties<C> ());

          void
          _parse (xercesc::InputSource const&,
                  std::basic_string<C> const& root_element_namespace,
                  std::basic_string<C> const& root_element_name,
                  xercesc::ErrorHandler&,
                  flags = 0,
                  properties<C> const& = properties<C> ());

          void
          _parse (xercesc::InputSource const&,
                  std::basic_string<C> const& root_element_namespace,
                  std::basic_string<C> const& root_element_name,
                  xercesc::SAX2XMLReader&,
                  flags = 0,
                  properties<C> const& = properties<C> ());
        };


        //
        //
        template <typename C>
        struct event_router: xercesc::DefaultHandler
        {
          typedef std::basic_string<C> string;

          event_router (traverser_base<C>& t)
              : t_ (t)
          {
          }

          // I know, some of those consts are stupid. But that's what
          // Xerces folks put into their interfaces and VC-7.1 thinks
          // there are different signatures if one strips this fluff off.
          //

          virtual void
          startElement(XMLCh const* const uri,
                       XMLCh const* const lname,
                       XMLCh const* const qname,
                       xercesc::Attributes const& attributes);

          virtual void
          endElement(XMLCh const* const uri,
                     XMLCh const* const lname,
                     XMLCh const* const qname);

          virtual void
          characters (XMLCh const* const s, unsigned int const length);

        private:
          traverser_base<C>& t_;
          string buf_;
        };
      }
    }
  }
}

#include <xsd/cxx/parser/xerces/elements.txx>

#endif  // XSD_CXX_PARSER_XERCES_ELEMENTS_HXX
