// file      : xsd/cxx/xml/dom/elements.txx
// author    : Boris Kolpackov <boris@codesynthesis.com>
// copyright : Copyright (c) 2005-2006 Code Synthesis Tools CC
// license   : GNU GPL v2 + exceptions; see accompanying LICENSE file

#include <xercesc/util/XMLUniDefs.hpp> // chLatin_L, etc

#include <xsd/cxx/xml/dom/bits/error-handler-proxy.hxx>

namespace xsd
{
  namespace cxx
  {
    namespace xml
    {
      namespace dom
      {
        template <typename C>
        xml::dom::auto_ptr<xercesc::DOMDocument>
        dom (xercesc::DOMInputSource const& is,
             error_handler<C>& eh,
             properties<C> const& prop,
             bool validate)
        {
          bits::error_handler_proxy<C> ehp (eh);
          return xml::dom::dom (is, ehp, prop, validate);
        }

        template <typename C>
        auto_ptr<xercesc::DOMDocument>
        dom (xercesc::DOMInputSource const& is,
             xercesc::DOMErrorHandler& eh,
             properties<C> const& prop,
             bool validate)
        {
          // HP aCC cannot handle using namespace xercesc;
          //
          using xercesc::DOMImplementationRegistry;
          using xercesc::DOMImplementationLS;
          using xercesc::DOMDocument;
          using xercesc::DOMBuilder;
          using xercesc::XMLUni;


          // Instantiate the DOM parser.
          //
          XMLCh const gLS[] = {xercesc::chLatin_L,
                               xercesc::chLatin_S,
                               xercesc::chNull};

          // Get an implementation of the Load-Store (LS) interface.
          //
          DOMImplementationLS* impl (
            static_cast<DOMImplementationLS*>(
              DOMImplementationRegistry::getDOMImplementation(gLS)));

          // Create a DOMBuilder.
          //
          auto_ptr<DOMBuilder> parser (
            impl->createDOMBuilder(DOMImplementationLS::MODE_SYNCHRONOUS, 0));

          // Discard comment nodes in the document.
          //
          parser->setFeature (XMLUni::fgDOMComments, false);

          // Disable datatype normalization. The XML 1.0 attribute value
          // normalization always occurs though.
          //
          parser->setFeature (XMLUni::fgDOMDatatypeNormalization, true);

          // Do not create EntityReference nodes in the DOM tree. No
          // EntityReference nodes will be created, only the nodes
          // corresponding to their fully expanded substitution text will be
          // created.
          //
          parser->setFeature (XMLUni::fgDOMEntities, false);

          // Perform Namespace processing.
          //
          parser->setFeature (XMLUni::fgDOMNamespaces, true);

          // Do not include ignorable whitespace in the DOM tree.
          //
          parser->setFeature (XMLUni::fgDOMWhitespaceInElementContent, false);

          if (validate)
          {
            parser->setFeature (XMLUni::fgDOMValidation, true);
            parser->setFeature (XMLUni::fgXercesSchema, true);
            parser->setFeature (XMLUni::fgXercesSchemaFullChecking, true);
            //parser->setFeature (XMLUni::fgXercesDOMHasPSVIInfo, true);
          }
          else
          {
            parser->setFeature (XMLUni::fgDOMValidation, false);
            parser->setFeature (XMLUni::fgXercesSchema, false);
            parser->setFeature (XMLUni::fgXercesSchemaFullChecking, false);
            //parser->setFeature (XMLUni::fgXercesDOMHasPSVIInfo, false);
          }

          // We will release DOM ourselves.
          //
          parser->setFeature (XMLUni::fgXercesUserAdoptsDOMDocument, true);


          // Transfer properies if any.
          //

          if (!prop.schema_location ().empty ())
          {
            xml::string sl (prop.schema_location ());
            void const* v (sl.c_str ());

            parser->setProperty (
              XMLUni::fgXercesSchemaExternalSchemaLocation,
              const_cast<void*> (v));
          }

          if (!prop.no_namespace_schema_location ().empty ())
          {
            xml::string sl (prop.no_namespace_schema_location ());
            void const* v (sl.c_str ());

            parser->setProperty (
              XMLUni::fgXercesSchemaExternalNoNameSpaceSchemaLocation,
              const_cast<void*> (v));
          }

          // Set error handler.
          //
          bits::error_handler_proxy<C> ehp (eh);
          parser->setErrorHandler (&ehp);

          auto_ptr<DOMDocument> doc (parser->parse (is));

          if (ehp.failed ())
            doc.reset ();

          return doc;
        }

        template <typename C>
        xml::dom::auto_ptr<xercesc::DOMDocument>
        dom (std::basic_string<C> const& uri,
             error_handler<C>& eh,
             properties<C> const& prop,
             bool validate)
        {
          bits::error_handler_proxy<C> ehp (eh);
          return xml::dom::dom (uri, ehp, prop, validate);
        }

        template <typename C>
        auto_ptr<xercesc::DOMDocument>
        dom (std::basic_string<C> const& uri,
             xercesc::DOMErrorHandler& eh,
             properties<C> const& prop,
             bool validate)
        {
          // HP aCC cannot handle using namespace xercesc;
          //
          using xercesc::DOMImplementationRegistry;
          using xercesc::DOMImplementationLS;
          using xercesc::DOMDocument;
          using xercesc::DOMBuilder;
          using xercesc::XMLUni;


          // Instantiate the DOM parser.
          //
          XMLCh const gLS[] = {xercesc::chLatin_L,
                               xercesc::chLatin_S,
                               xercesc::chNull};

          // Get an implementation of the Load-Store (LS) interface.
          //
          DOMImplementationLS* impl (
            static_cast<DOMImplementationLS*>(
              DOMImplementationRegistry::getDOMImplementation(gLS)));

          // Create a DOMBuilder.
          //
          auto_ptr<DOMBuilder> parser (
            impl->createDOMBuilder(DOMImplementationLS::MODE_SYNCHRONOUS, 0));

          // Discard comment nodes in the document.
          //
          parser->setFeature (XMLUni::fgDOMComments, false);

          // Disable datatype normalization. The XML 1.0 attribute value
          // normalization always occurs though.
          //
          parser->setFeature (XMLUni::fgDOMDatatypeNormalization, true);

          // Do not create EntityReference nodes in the DOM tree. No
          // EntityReference nodes will be created, only the nodes
          // corresponding to their fully expanded substitution text will be
          // created.
          //
          parser->setFeature (XMLUni::fgDOMEntities, false);

          // Perform Namespace processing.
          //
          parser->setFeature (XMLUni::fgDOMNamespaces, true);

          // Do not include ignorable whitespace in the DOM tree.
          //
          parser->setFeature (XMLUni::fgDOMWhitespaceInElementContent, false);

          if (validate)
          {
            parser->setFeature (XMLUni::fgDOMValidation, true);
            parser->setFeature (XMLUni::fgXercesSchema, true);
            parser->setFeature (XMLUni::fgXercesSchemaFullChecking, true);
            //parser->setFeature (XMLUni::fgXercesDOMHasPSVIInfo, true);
          }
          else
          {
            parser->setFeature (XMLUni::fgDOMValidation, false);
            parser->setFeature (XMLUni::fgXercesSchema, false);
            parser->setFeature (XMLUni::fgXercesSchemaFullChecking, false);
            //parser->setFeature (XMLUni::fgXercesDOMHasPSVIInfo, false);
          }

          // We will release DOM ourselves.
          //
          parser->setFeature (XMLUni::fgXercesUserAdoptsDOMDocument, true);

          // Transfer properies if any.
          //

          if (!prop.schema_location ().empty ())
          {
            xml::string sl (prop.schema_location ());
            void const* v (sl.c_str ());

            parser->setProperty (
              XMLUni::fgXercesSchemaExternalSchemaLocation,
              const_cast<void*> (v));
          }

          if (!prop.no_namespace_schema_location ().empty ())
          {
            xml::string sl (prop.no_namespace_schema_location ());
            void const* v (sl.c_str ());

            parser->setProperty (
              XMLUni::fgXercesSchemaExternalNoNameSpaceSchemaLocation,
              const_cast<void*> (v));
          }

          // Set error handler.
          //
          bits::error_handler_proxy<C> ehp (eh);
          parser->setErrorHandler (&ehp);

          auto_ptr<DOMDocument> doc (
            parser->parseURI (string (uri).c_str ()));

          if (ehp.failed ())
            doc.reset ();

          return doc;
        }
      }
    }
  }
}
